// src/app/controllers/api/app/attachments.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import crypto from "crypto";
import * as fsp from "fs/promises";
import * as fs from "fs";
import { withRlsTx, type Db } from "@/lib/postgres.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";
import path from "path";

const fixUtf8FromLatin1 = (s: string | undefined | null) => {
  if (!s) return s ?? "";
  // Mojibake tipikus jelei (UTF-8 byte-ok latin1-ként olvasva)
  if (!/Ã.|Â.|Å.|Ð.|Ñ.|Ø.|Þ./.test(s)) return s;

  try {
    return Buffer.from(s, "latin1").toString("utf8");
  } catch {
    return s;
  }
};
function sanitizeAsciiFallback(name: string) {
  // header-safe ASCII fallback (ne legyen idézőjel, backslash, vezérlő karakter)
  return (
    (name || "file")
      .replace(/[^\x20-\x7E]/g, "_") // non-ascii -> _
      .replace(/["\\]/g, "_")
      .replace(/[<>:/\\|?*\u0000-\u001F]/g, "_")
      .trim() || "file"
  );
}
async function resolveMeetingId(db: Db, meetingRef: string): Promise<string> {
  // uuid → id
  if (/^[0-9a-fA-F-]{36}$/.test(meetingRef)) return meetingRef;

  const { rows } = await db.query<{ id: string }>(
    `SELECT id FROM public.meetings WHERE code = $1 LIMIT 1`,
    [meetingRef]
  );

  if (!rows.length) {
    const e: any = new Error("Meeting not found");
    e.statusCode = 404;
    throw e;
  }
  return rows[0].id;
}

function encodeRFC5987ValueChars(str: string) {
  // RFC 5987: UTF-8 + percent-encode
  return encodeURIComponent(str)
    .replace(/['()]/g, escape)
    .replace(/\*/g, "%2A")
    .replace(/%(7C|60|5E)/g, (m) => m.toLowerCase());
}

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}
const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

async function resolveThreadId(db: Db, threadRef: string): Promise<string> {
  if (UUID_RE.test(threadRef)) return z.string().uuid().parse(threadRef);

  const r = await db.query(
    `SELECT id FROM public.case_threads WHERE code=$1 LIMIT 1`,
    [threadRef.trim()]
  );

  if (!r.rows?.length) {
    const e: any = new Error("Thread not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}

async function sha256File(filePath: string): Promise<string> {
  const buf = await fsp.readFile(filePath);
  return crypto.createHash("sha256").update(buf).digest("hex");
}

function getUploadRootAbs() {
  // állítható env-ből, de default is oké
  const root =
    process.env.UPLOAD_ROOT || process.env.QMS_UPLOAD_ROOT || "uploads";
  return path.resolve(process.cwd(), root);
}

function toStoredPath(filePath: string) {
  // multer adhat relatívat vagy abszolútot -> egységesítjük
  const uploadRootAbs = getUploadRootAbs();
  const abs = path.resolve(filePath);

  // csak az upload root-on belül engedjük
  const rel = path.relative(uploadRootAbs, abs);
  if (!rel || rel.startsWith("..") || path.isAbsolute(rel)) {
    const e: any = new Error("Invalid upload path (outside upload root)");
    e.statusCode = 400;
    throw e;
  }
  // DB-be mindig POSIX-szerűt tároljunk (cross-platform barát)
  return rel.split(path.sep).join("/");
}

function fromStoredPath(storedPath: string) {
  const uploadRootAbs = getUploadRootAbs();

  // legacy: ha valamiért abszolút volt DB-ben, akkor is védd
  const abs = path.isAbsolute(storedPath)
    ? path.resolve(storedPath)
    : path.resolve(uploadRootAbs, storedPath);

  const rel = path.relative(uploadRootAbs, abs);
  if (!rel || rel.startsWith("..") || path.isAbsolute(rel)) {
    const e: any = new Error("Not found");
    e.statusCode = 404;
    throw e;
  }
  return abs;
}

export async function uploadThreadAttachment(req: Request, res: Response) {
  try {
    const threadRef = z
      .string()
      .min(1)
      .parse((req.params as any).threadRef);
    const f = (req as any).file as Express.Multer.File | undefined;

    if (!f) {
      return res
        .status(400)
        .json({ error: "No file uploaded (or rejected by filter)" });
    }

    // ✅ EZ AZ: DB-be már a javított nevet mentsd
    const originalNameForDb = fixUtf8FromLatin1(f.originalname);

    const relPath = toStoredPath(f.path);
    const hash = await sha256File(f.path);

    const result = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadId = await resolveThreadId(db, threadRef);

      const { rows: t } = await db.query<{ case_id: string }>(
        `SELECT case_id FROM public.case_threads WHERE id=$1`,
        [threadId]
      );
      if (!t.length) {
        const e: any = new Error("Thread not found");
        e.statusCode = 404;
        throw e;
      }
      const caseId = t[0].case_id;

      // ✅ Permission check: user must be active case member (or have sensitive access)
      const { rows: mem } = await db.query<{ ok: boolean }>(
        `
  SELECT EXISTS (
    SELECT 1
    FROM public.case_members m
    WHERE m.case_id = $1
      AND m.user_id = public.app_user_id()
      AND m.is_active = true
  ) AS ok
  `,
        [caseId]
      );

      if (!mem[0]?.ok) {
        const { rows: s } = await db.query<{ ok: boolean }>(
          `SELECT public.app_has_sensitive_access() AS ok`
        );

        if (!s[0]?.ok) {
          const e: any = new Error("Not allowed to attach to this thread");
          e.statusCode = 403;
          throw e;
        }
      }

      const { rows: a } = await db.query<{ id: string }>(
        `
        INSERT INTO public.attachments
          (id, storage, path, original_name, mime_type, size_bytes, sha256, uploaded_by)
        VALUES
          (gen_random_uuid(), 'fs', $1, $2, $3, $4, $5, $6)
        RETURNING id
        `,
        [
          relPath,
          originalNameForDb, // ✅ itt volt f.originalname
          f.mimetype ?? null,
          f.size ?? null,
          hash,
          req.user!.id,
        ]
      );

      const attachmentId = a[0].id;

      await db.query(
        `INSERT INTO public.thread_attachments (thread_id, attachment_id)
         VALUES ($1,$2) ON CONFLICT DO NOTHING`,
        [threadId, attachmentId]
      );

      return { attachmentId, caseId, threadId };
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "ATTACHMENT_UPLOADED",
      objectType: "attachment",
      objectId: result.attachmentId,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {
        thread_ref: threadRef,
        thread_id: result.threadId,
        case_id: result.caseId,
        // ✅ audit logban is a javítottat érdemes
        file: { name: originalNameForDb, mime: f.mimetype, size: f.size },
      },
    });

    res.status(201).json({ id: result.attachmentId });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "uploadThreadAttachment", message: e.message });
  }
}

// --- MEETING attachment upload ---
export async function uploadMeetingAttachment(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);

    const f = (req as any).file as Express.Multer.File | undefined;

    if (!f) {
      return res
        .status(400)
        .json({ error: "No file uploaded (or rejected by filter)" });
    }

    // ✅ DB-be már a javított nevet mentsd
    const originalNameForDb = fixUtf8FromLatin1(f.originalname);

    const relPath = toStoredPath(f.path);

    const hash = await sha256File(f.path);

    const result = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      // jogosultság: legyen legalább 1 olyan agenda item a meetingben,
      // amelynek case-éhez a user tag (RLS + explicit check)
      const { rows: ok } = await db.query<{ ok: boolean }>(
        `
        SELECT EXISTS (
          SELECT 1
          FROM public.meeting_agenda_items ai
          JOIN public.case_members m
            ON m.case_id = ai.case_id
          WHERE ai.meeting_id = $1
            AND m.user_id = public.app_user_id()
            AND m.is_active = true
        ) AS ok
        `,
        [meetingId]
      );

      if (!ok[0]?.ok) {
        const e: any = new Error("Not allowed to attach to this meeting");
        e.statusCode = 403;
        throw e;
      }

      const { rows: a } = await db.query<{ id: string }>(
        `
        INSERT INTO public.attachments
          (id, storage, path, original_name, mime_type, size_bytes, sha256, uploaded_by)
        VALUES
          (gen_random_uuid(), 'fs', $1, $2, $3, $4, $5, $6)
        RETURNING id
        `,
        [
          relPath,
          originalNameForDb,
          f.mimetype ?? null,
          f.size ?? null,
          hash,
          req.user!.id,
        ]
      );

      const attachmentId = a[0].id;

      await db.query(
        `
        INSERT INTO public.meeting_attachments (meeting_id, attachment_id)
        VALUES ($1,$2)
        ON CONFLICT DO NOTHING
        `,
        [meetingId, attachmentId]
      );

      return { attachmentId, meetingId };
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_ATTACHMENT_UPLOADED",
      objectType: "attachment",
      objectId: result.attachmentId,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {
        meeting_id: result.meetingId,
        meeting_ref: meetingRef,
        file: { name: originalNameForDb, mime: f.mimetype, size: f.size },
      },
    });

    res.status(201).json({ id: result.attachmentId });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "uploadMeetingAttachment", message: e.message });
  }
}

/**
 * jogosultság ellenőrzés:
 * - thread_attachments -> thread -> case_members
 * - case_attachments -> case_members
 * - meeting_attachments -> agenda_items -> case_members
 * - decision_attachments -> decision.case_id -> case_members
 * plusz: super-admin (app_has_sensitive_access)
 */
async function resolveAttachmentForUser(db: Db, attachmentId: string) {
  const { rows } = await db.query<{
    id: string;
    path: string;
    original_name: string;
    mime_type: string | null;
    size_bytes: string | number | null;
  }>(
    `
    WITH allowed AS (
      -- 1) thread attachments
      SELECT a.id, a.path, a.original_name, a.mime_type, a.size_bytes
      FROM public.attachments a
      JOIN public.thread_attachments ta ON ta.attachment_id = a.id
      JOIN public.case_threads t ON t.id = ta.thread_id
      JOIN public.case_members m ON m.case_id = t.case_id
      WHERE a.id = $1
        AND m.user_id = public.app_user_id()
        AND m.is_active = true

      UNION ALL

      -- 2) case attachments
      SELECT a.id, a.path, a.original_name, a.mime_type, a.size_bytes
      FROM public.attachments a
      JOIN public.case_attachments ca ON ca.attachment_id = a.id
      JOIN public.case_members m ON m.case_id = ca.case_id
      WHERE a.id = $1
        AND m.user_id = public.app_user_id()
        AND m.is_active = true

      UNION ALL

      -- 3) meeting attachments -> meeting agenda -> case_members
      SELECT a.id, a.path, a.original_name, a.mime_type, a.size_bytes
      FROM public.attachments a
      JOIN public.meeting_attachments ma ON ma.attachment_id = a.id
      JOIN public.meeting_agenda_items ai ON ai.meeting_id = ma.meeting_id
      JOIN public.case_members m ON m.case_id = ai.case_id
      WHERE a.id = $1
        AND m.user_id = public.app_user_id()
        AND m.is_active = true

      UNION ALL

      -- 4) decision attachments -> decision.case_id -> case_members
      SELECT a.id, a.path, a.original_name, a.mime_type, a.size_bytes
      FROM public.attachments a
      JOIN public.decision_attachments da ON da.attachment_id = a.id
      JOIN public.decisions d ON d.id = da.decision_id
      JOIN public.case_members m ON m.case_id = d.case_id
      WHERE a.id = $1
        AND m.user_id = public.app_user_id()
        AND m.is_active = true
    )
    SELECT * FROM allowed
    LIMIT 1
    `,
    [attachmentId]
  );

  if (!rows.length) {
    const { rows: s } = await db.query<{ ok: boolean }>(
      `SELECT public.app_has_sensitive_access() AS ok`
    );

    if (s[0]?.ok === true) {
      const { rows: a } = await db.query<{
        id: string;
        path: string;
        original_name: string;
        mime_type: string | null;
        size_bytes: string | number | null;
      }>(
        `
        SELECT id, path, original_name, mime_type, size_bytes
        FROM public.attachments
        WHERE id=$1
        LIMIT 1
        `,
        [attachmentId]
      );
      return a[0] ?? null;
    }
  }

  return rows[0] ?? null;
}

function safeFilename(name: string): string {
  const base = name?.trim() || "file";
  return base.replace(/[/\\?%*:|"<>]/g, "_");
}

function parseRange(
  rangeHeader: string,
  size: number
): { start: number; end: number } | null {
  const m = /^bytes=(\d*)-(\d*)$/i.exec(rangeHeader.trim());
  if (!m) return null;

  const startStr = m[1];
  const endStr = m[2];

  let start: number;
  let end: number;

  if (startStr === "" && endStr === "") return null;

  if (startStr === "") {
    const suffixLen = Number(endStr);
    if (!Number.isFinite(suffixLen) || suffixLen <= 0) return null;
    start = Math.max(size - suffixLen, 0);
    end = size - 1;
    return { start, end };
  }

  start = Number(startStr);
  if (!Number.isFinite(start) || start < 0) return null;

  if (endStr === "") {
    end = size - 1;
  } else {
    end = Number(endStr);
    if (!Number.isFinite(end) || end < 0) return null;
  }

  if (start > end) return null;
  if (start >= size) return null;

  end = Math.min(end, size - 1);
  return { start, end };
}

async function streamAttachment(
  req: Request,
  res: Response,
  opts: { isHead: boolean; disposition: "attachment" | "inline" }
) {
  const attachmentId = z.string().uuid().parse(req.params.attachmentId);

  const record = await withRlsTx(rlsCtx(req), async (db: Db) => {
    return resolveAttachmentForUser(db, attachmentId);
  });

  if (!record) return res.status(404).json({ error: "Not found" });

  const filePath = fromStoredPath(record.path);

  let stat;
  try {
    stat = await fsp.stat(filePath);
  } catch (err: any) {
    if (err?.code === "ENOENT") {
      return res.status(404).json({ error: "Not found" });
    }
    throw err;
  }
  const size = stat.size;
  const mime = record.mime_type || "application/octet-stream";
  const originalName = (record.original_name || "file").trim() || "file";

  const asciiFallback = sanitizeAsciiFallback(originalName);
  const utf8Name = encodeRFC5987ValueChars(originalName);

  res.setHeader("Accept-Ranges", "bytes");
  res.setHeader("Content-Type", mime);
  res.setHeader("Cache-Control", "private, no-store");

  res.setHeader(
    "Content-Disposition",
    `${opts.disposition}; filename="${asciiFallback}"; filename*=UTF-8''${utf8Name}`
  );

  if (opts.isHead) {
    res.setHeader("Content-Length", String(size));
    return res.status(200).end();
  }

  const rangeHeader = req.headers.range;

  // Full
  if (!rangeHeader) {
    res.setHeader("Content-Length", String(size));
    const stream = fs.createReadStream(filePath);
    stream.on("error", () =>
      res.headersSent ? res.end() : res.status(500).end()
    );
    return stream.pipe(res);
  }

  // Partial
  const r = parseRange(String(rangeHeader), size);
  if (!r) {
    res.setHeader("Content-Range", `bytes */${size}`);
    return res.status(416).end();
  }

  const { start, end } = r;
  const chunkSize = end - start + 1;

  res.status(206);
  res.setHeader("Content-Length", String(chunkSize));
  res.setHeader("Content-Range", `bytes ${start}-${end}/${size}`);

  const stream = fs.createReadStream(filePath, { start, end });
  stream.on("error", () =>
    res.headersSent ? res.end() : res.status(500).end()
  );
  return stream.pipe(res);
}

export async function downloadAttachment(req: Request, res: Response) {
  try {
    return await streamAttachment(req, res, {
      isHead: false,
      disposition: "attachment",
    });
  } catch (e: any) {
    return res
      .status(e.statusCode ?? 400)
      .json({ error: "downloadAttachment", message: e.message });
  }
}

export async function headAttachment(req: Request, res: Response) {
  try {
    return await streamAttachment(req, res, {
      isHead: true,
      disposition: "attachment",
    });
  } catch (e: any) {
    return res
      .status(e.statusCode ?? 400)
      .json({ error: "headAttachment", message: e.message });
  }
}

// ✅ PREVIEW: inline stream
export async function previewAttachment(req: Request, res: Response) {
  try {
    return await streamAttachment(req, res, {
      isHead: false,
      disposition: "inline",
    });
  } catch (e: any) {
    return res
      .status(e.statusCode ?? 400)
      .json({ error: "previewAttachment", message: e.message });
  }
}

export async function headPreviewAttachment(req: Request, res: Response) {
  try {
    return await streamAttachment(req, res, {
      isHead: true,
      disposition: "inline",
    });
  } catch (e: any) {
    return res
      .status(e.statusCode ?? 400)
      .json({ error: "headPreviewAttachment", message: e.message });
  }
}
export async function deleteAttachment(req: Request, res: Response) {
  try {
    const attachmentId = z.string().uuid().parse(req.params.attachmentId);

    const result = await withRlsTx(rlsCtx(req), async (db: Db) => {
      // 1) jogosultság: ugyanaz, mint preview/download
      const record = await resolveAttachmentForUser(db, attachmentId);
      if (!record) {
        const e: any = new Error("Not found");
        e.statusCode = 404;
        throw e;
      }

      // 2) töröljük a "linkeket" (ahol létezhet)
      // ha nálatok csak thread + meeting kell, a többieket nyugodtan kiszedheted
      await db.query(
        `DELETE FROM public.thread_attachments WHERE attachment_id=$1`,
        [attachmentId]
      );
      await db.query(
        `DELETE FROM public.meeting_attachments WHERE attachment_id=$1`,
        [attachmentId]
      );
      await db.query(
        `DELETE FROM public.case_attachments WHERE attachment_id=$1`,
        [attachmentId]
      );
      await db.query(
        `DELETE FROM public.decision_attachments WHERE attachment_id=$1`,
        [attachmentId]
      );

      // 3) megnézzük maradt-e bárhol referencia
      const { rows: refs } = await db.query<{ n: number }>(
        `
        SELECT
          (
            (SELECT COUNT(*) FROM public.thread_attachments WHERE attachment_id=$1) +
            (SELECT COUNT(*) FROM public.meeting_attachments WHERE attachment_id=$1) +
            (SELECT COUNT(*) FROM public.case_attachments WHERE attachment_id=$1) +
            (SELECT COUNT(*) FROM public.decision_attachments WHERE attachment_id=$1)
          )::int AS n
        `,
        [attachmentId]
      );

      const stillReferenced = (refs[0]?.n ?? 0) > 0;

      // 4) ha nincs referencia, töröljük a sor + fájl
      if (!stillReferenced) {
        // újra lekérjük a path-ot, mert a record kelleni fog fájltörléshez
        const { rows: a } = await db.query<{
          path: string;
          original_name: string | null;
        }>(
          `SELECT path, original_name FROM public.attachments WHERE id=$1 LIMIT 1`,
          [attachmentId]
        );
        const pathToDelete = a[0]?.path;

        await db.query(`DELETE FROM public.attachments WHERE id=$1`, [
          attachmentId,
        ]);

        // fájl törlés nem DB-n belül kötelező, de praktikus
        if (pathToDelete) {
          try {
            await fsp.unlink(fromStoredPath(pathToDelete));
          } catch (err: any) {
            // ha már nincs ott, oké; mást logolhatunk
            if (err?.code !== "ENOENT") throw err;
          }
        }
      }

      return { deleted: !stillReferenced, unlinked: true };
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "ATTACHMENT_DELETED",
      objectType: "attachment",
      objectId: req.params.attachmentId,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { result },
    });

    return res.status(200).json(result);
  } catch (e: any) {
    return res
      .status(e.statusCode ?? 400)
      .json({ error: "deleteAttachment", message: e.message });
  }
}
// export async function removeThreadAttachment(req: Request, res: Response) {
//   try {
//     const attachmentId = z.string().uuid().parse(req.params.attachmentId);
//     const threadRef = z.string().min(1).parse(req.params.threadRef);

//     const result = await withRlsTx(rlsCtx(req), async (db: Db) => {
//       const threadId = await resolveThreadId(db, threadRef);

//       // permission check: must be able to access this attachment (same as preview)
//       const record = await resolveAttachmentForUser(db, attachmentId);
//       if (!record) {
//         const e: any = new Error("Not found");
//         e.statusCode = 404;
//         throw e;
//       }

//       // unlink ONLY from this thread
//       await db.query(
//         `DELETE FROM public.thread_attachments WHERE thread_id=$1 AND attachment_id=$2`,
//         [threadId, attachmentId]
//       );
//       console.log("[removeFromThread]", {
//         threadRef,
//         resolvedThreadId: threadId,
//         attachmentId,
//       });
//       return { unlinked: true };
//     });

//     return res.status(200).json(result);
//   } catch (e: any) {
//     return res
//       .status(e.statusCode ?? 400)
//       .json({ error: "removeThreadAttachment", message: e.message });
//   }
// }
export async function removeThreadAttachment(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);
    const attachmentId = z.string().uuid().parse(req.params.attachmentId);

    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadId = await resolveThreadId(db, threadRef);

      console.log("[removeFromThread]", {
        threadRef,
        resolvedThreadId: threadId,
        attachmentId,
      });

      // permission (optional but recommended)
      const record = await resolveAttachmentForUser(db, attachmentId);
      if (!record) {
        const e: any = new Error("Not found");
        e.statusCode = 404;
        throw e;
      }

      const del = await db.query(
        `DELETE FROM public.thread_attachments
         WHERE thread_id=$1 AND attachment_id=$2`,
        [threadId, attachmentId]
      );

      const { rows: chk } = await db.query<{ still: boolean }>(
        `SELECT EXISTS(
           SELECT 1 FROM public.thread_attachments
           WHERE thread_id=$1 AND attachment_id=$2
         ) AS still`,
        [threadId, attachmentId]
      );

      return {
        unlinked: del.rowCount > 0,
        deletedLinks: del.rowCount,
        stillLinked: chk[0]?.still ?? null,
        threadId,
      };
    });

    return res.status(200).json(out);
  } catch (e: any) {
    return res
      .status(e.statusCode ?? 400)
      .json({ error: "removeThreadAttachment", message: e.message });
  }
}
