// src/app/controllers/api/admin/case-types/caseTypes.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

function requireSuperAdmin(req: Request) {
  if (req.user?.accessLevel?.name !== "super-admin") {
    const e: any = new Error("Forbidden");
    e.statusCode = 403;
    throw e;
  }
}

const ListQuery = z.object({
  includeInactive: z.union([z.literal("1"), z.literal("true")]).optional(),
  q: z.string().optional(),
});

export async function listCaseTypes(req: Request, res: Response) {
  try {
    const q = ListQuery.parse(req.query);

    const rows = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const includeInactive =
        q.includeInactive === "1" || q.includeInactive === "true";
      const needle = (q.q ?? "").trim();

      const params: any[] = [];
      const where: string[] = [];

      if (!includeInactive) where.push(`is_active = true`);

      if (needle) {
        params.push(`%${needle}%`);
        where.push(
          `(key ILIKE $${params.length} OR name ILIKE $${params.length})`
        );
      }

      const sql = `
        SELECT id, key, name, description, is_active, created_at, updated_at
        FROM public.case_types
        ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
        ORDER BY is_active DESC, name ASC
      `;
      const r = await db.query(sql, params);
      return r.rows;
    });

    res.json({ caseTypes: rows });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listCaseTypes", message: e.message });
  }
}

const CreateBody = z.object({
  key: z.string().min(2),
  name: z.string().min(2),
  description: z.string().optional(),
});

export async function createCaseType(req: Request, res: Response) {
  try {
    requireSuperAdmin(req);
    const b = CreateBody.parse(req.body);

    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const key = b.key.trim().toLowerCase();
      const name = b.name.trim();

      const r = await db.query(
        `
        INSERT INTO public.case_types (id, key, name, description, is_active)
        VALUES (gen_random_uuid(), $1, $2, $3, true)
        RETURNING id, key, name, description, is_active, created_at, updated_at
        `,
        [key, name, b.description ?? null]
      );
      return r.rows[0];
    });

    res.status(201).json({ caseType: out });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createCaseType", message: e.message });
  }
}

const UpdateBody = z.object({
  name: z.string().min(2).optional(),
  description: z.string().nullable().optional(),
  isActive: z.boolean().optional(),
});

export async function updateCaseType(req: Request, res: Response) {
  try {
    requireSuperAdmin(req);

    const id = z.string().uuid().parse(req.params.caseTypeId);
    const b = UpdateBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const sets: string[] = [];
      const params: any[] = [id];

      const push = (sql: string, val: any) => {
        params.push(val);
        sets.push(sql.replace("$X", `$${params.length}`));
      };

      if (b.name !== undefined) push(`name=$X`, b.name.trim());
      if (b.description !== undefined)
        push(`description=$X`, b.description ?? null);
      if (b.isActive !== undefined) push(`is_active=$X`, b.isActive);

      if (!sets.length) return;

      await db.query(
        `UPDATE public.case_types SET ${sets.join(", ")} WHERE id=$1`,
        params
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "updateCaseType", message: e.message });
  }
}
