import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, Db } from "@/lib/postgres.js";
import { requireCaseRole } from "../../../../utils/aclCase.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";
// import { resolveCaseId } from "../../../../utils/resolveCaseId.js";
const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

async function resolveCaseId(db: any, caseRef: string): Promise<string> {
  if (UUID_RE.test(caseRef)) return z.string().uuid().parse(caseRef);

  const r = await db.query(
    `SELECT id FROM public.cases WHERE code=$1 LIMIT 1`,
    [caseRef.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Case not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}
function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}
const BoolQuery = z
  .union([
    z.literal("1"),
    z.literal("0"),
    z.literal("true"),
    z.literal("false"),
  ])
  .optional()
  .transform((v) => v === "1" || v === "true");

const CaseQuery = z.object({
  status: z.string().optional(),
  mine: z.string().optional(),
  q: z.string().optional(),
  includeArchived: BoolQuery, // boolean lesz: true/false
});
export async function listCases(req: Request, res: Response) {
  try {
    const q = CaseQuery.parse(req.query);

    const data = await withRlsTx(rlsCtx(req), async (db) => {
      const params: any[] = [];
      const where: string[] = [];

      if (!q.includeArchived) {
        where.push(`c.archived_at IS NULL`);
      }

      if (q.status) {
        params.push(q.status);
        where.push(`c.status = $${params.length}::public.case_status`);
      }

      if (q.mine === "1") {
        where.push(`
          EXISTS (
            SELECT 1 FROM public.case_members m
            WHERE m.case_id=c.id AND m.user_id=public.app_user_id() AND m.is_active=true
          )
        `);
      }

      if (q.q?.trim()) {
        params.push(`%${q.q.trim()}%`);
        where.push(
          `(c.title ILIKE $${params.length} OR c.code ILIKE $${params.length})`
        );
      }

      const sql = `
        SELECT
          c.id,
          c.code,
          c.title,
          c.description,
          c.case_type,
          c.flow_type,
          c.playbook_id,
          pb.key   AS playbook_key,
          pb.name  AS playbook_name,
          c.status,
          c.priority,
          c.owner_user_id,
          ou.name     AS owner_name,
          ou.email    AS owner_email,
          ou.username AS owner_username,
          c.due_at,
          c.created_at,
          c.updated_at,
          c.archived_at,
          c.archived_by
        FROM public.cases c
        LEFT JOIN public.users ou ON ou.id = c.owner_user_id
        LEFT JOIN public.playbooks pb ON pb.id = c.playbook_id
        LEFT JOIN public.case_board_positions bp
          ON bp.case_id = c.id
         AND bp.status = c.status
        ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
        ORDER BY
          CASE c.status
            WHEN 'needs_decision' THEN 0
            WHEN 'blocked' THEN 1
            WHEN 'in_progress' THEN 2
            WHEN 'new' THEN 3
            WHEN 'closed' THEN 4
            ELSE 9
          END,
          COALESCE(bp.position, 999999999) ASC,
          c.updated_at DESC
        LIMIT 200
      `;

      const { rows } = await db.query(sql, params);
      return rows;
    });

    res.json({ cases: data });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listCases", message: e.message });
  }
}

// export async function listCases(req: Request, res: Response) {
//   try {
//     const q = CaseQuery.parse(req.query);

//     const data = await withRlsTx(rlsCtx(req), async (db) => {
//       const params: any[] = [];
//       const where: string[] = [];

//       if (!q.includeArchived) {
//         where.push(`c.archived_at IS NULL`);
//       }

//       if (q.status) {
//         params.push(q.status);
//         where.push(`c.status = $${params.length}::public.case_status`);
//       }

//       if (q.mine === "1") {
//         where.push(`
//           EXISTS (
//             SELECT 1 FROM public.case_members m
//             WHERE m.case_id=c.id AND m.user_id=public.app_user_id() AND m.is_active=true
//           )
//         `);
//       }

//       if (q.q?.trim()) {
//         params.push(`%${q.q.trim()}%`);
//         where.push(
//           `(c.title ILIKE $${params.length} OR c.code ILIKE $${params.length})`
//         );
//       }

//       const sql = `
//         SELECT
//           c.id, c.code, c.title, c.description,
//           c.case_type,
//           c.flow_type,
//           c.playbook_id,
//           pb.name AS playbook_name,
//           pb.key  AS playbook_key,
//           c.status, c.priority,
//           c.owner_user_id,
//           ou.name     AS owner_name,
//           ou.email    AS owner_email,
//           ou.username AS owner_username,
//           c.due_at, c.created_at, c.updated_at,
//           c.archived_at, c.archived_by
//         FROM public.cases c
//         LEFT JOIN public.users ou ON ou.id = c.owner_user_id
//         LEFT JOIN public.playbooks pb ON pb.id = c.playbook_id
//         LEFT JOIN public.case_board_positions bp
//           ON bp.case_id = c.id
//          AND bp.status = c.status
//         ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
//         ORDER BY
//           CASE c.status
//             WHEN 'needs_decision' THEN 0
//             WHEN 'blocked' THEN 1
//             WHEN 'in_progress' THEN 2
//             WHEN 'new' THEN 3
//             WHEN 'closed' THEN 4
//             ELSE 9
//           END,
//           COALESCE(bp.position, 999999999) ASC,
//           c.updated_at DESC
//         LIMIT 200
//       `;

//       const { rows } = await db.query(sql, params);
//       return rows;
//     });

//     res.json({ cases: data });
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "listCases", message: e.message });
//   }
// }

const CreateCaseBody = z.object({
  title: z.string().min(2),
  description: z.string().optional(),
  caseType: z.string().optional(),
  flowType: z.enum(["transient", "continuous"]).optional(), // 👈 ADD
  playbookId: z.string().uuid().optional(), // 👈 ADD
  priority: z.number().int().min(1).max(5).optional(),
  dueAt: z.string().datetime().optional(),
});
export async function createCase(req: Request, res: Response) {
  try {
    const b = CreateCaseBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db) => {
      const { rows } = await db.query<{ id: string; code: string }>(
        `
        INSERT INTO public.cases (
  id, title, description, case_type, flow_type, playbook_id,
  priority, due_at, owner_user_id, created_by
)
VALUES (
  gen_random_uuid(), $1, $2, $3, COALESCE($4,'transient'), $5,
  COALESCE($6, 3), $7, $8, $8
)
RETURNING id, code

        `,
        [
          b.title.trim(),
          b.description ?? null,
          b.caseType ?? null,
          b.flowType ?? null,
          b.playbookId ?? null,
          b.priority ?? null,
          b.dueAt ? new Date(b.dueAt) : null,
          req.user!.id,
        ]
      );

      const caseId = rows[0].id;

      await db.query(
        `INSERT INTO public.case_members (case_id, user_id, role, is_active)
         VALUES ($1, $2, 'owner'::public.case_role, true)
         ON CONFLICT (case_id, user_id) DO UPDATE SET role=EXCLUDED.role, is_active=true`,
        [caseId, req.user!.id]
      );

      return rows[0];
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "CASE_CREATED",
      objectType: "case",
      objectId: created.id,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { title: b.title, case_type: b.caseType, code: created.code },
    });

    res.status(201).json({ id: created.id, code: created.code });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createCase", message: e.message });
  }
}
// export async function getCaseDetail(req: Request, res: Response) {
//   try {
//     const caseRef = z.string().min(1).parse(req.params.caseRef);

//     const data = await withRlsTx(rlsCtx(req), async (db) => {
//       const caseId = await resolveCaseId(db, caseRef);

//       const { rows: c } = await db.query(
//         `
//         SELECT
//           c.id, c.code, c.title, c.description,
//           c.case_type,
//           c.flow_type,
//           c.playbook_id,
//           pb.name AS playbook_name,
//           pb.key  AS playbook_key,
//           c.status, c.priority,
//           c.owner_user_id,
//           ou.name     AS owner_name,
//           ou.email    AS owner_email,
//           ou.username AS owner_username,
//           c.due_at, c.created_at, c.updated_at,
//           c.archived_at, c.archived_by
//         FROM public.cases c
//         LEFT JOIN public.users ou ON ou.id = c.owner_user_id
//         LEFT JOIN public.playbooks pb ON pb.id = c.playbook_id
//         WHERE c.id=$1
//         `,
//         [caseId]
//       );
//       if (!c.length) return null;

//       const { rows: members } = await db.query(
//         `
//         SELECT
//           m.user_id,
//           m.role,
//           m.is_active,
//           m.joined_at,
//           u.name     AS user_name,
//           u.email    AS user_email,
//           u.username AS user_username
//         FROM public.case_members m
//         JOIN public.users u ON u.id = m.user_id
//         WHERE m.case_id=$1
//         ORDER BY m.role, m.joined_at
//         `,
//         [caseId]
//       );

//       return { case: c[0], members };
//     });

//     if (!data) return res.status(404).json({ error: "Not found" });
//     res.json(data);
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "getCaseDetail", message: e.message });
//   }
// }
export async function getCaseDetail(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);

    const data = await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      const { rows: c } = await db.query(
        `
        SELECT
          c.id,
          c.code,
          c.title,
          c.description,
          c.case_type,
          c.flow_type,
          c.playbook_id,
          pb.key   AS playbook_key,
          pb.name  AS playbook_name,
          pb.description AS playbook_description,
          c.status,
          c.priority,
          c.owner_user_id,
          ou.name     AS owner_name,
          ou.email    AS owner_email,
          ou.username AS owner_username,
          c.due_at,
          c.created_at,
          c.updated_at,
          c.archived_at,
          c.archived_by
        FROM public.cases c
        LEFT JOIN public.users ou ON ou.id = c.owner_user_id
        LEFT JOIN public.playbooks pb ON pb.id = c.playbook_id
        WHERE c.id=$1
        `,
        [caseId]
      );
      if (!c.length) return null;

      const { rows: members } = await db.query(
        `
        SELECT
          m.user_id,
          m.role,
          m.is_active,
          m.joined_at,
          u.name     AS user_name,
          u.email    AS user_email,
          u.username AS user_username
        FROM public.case_members m
        JOIN public.users u ON u.id = m.user_id
        WHERE m.case_id=$1
        ORDER BY m.role, m.joined_at
        `,
        [caseId]
      );

      return { case: c[0], members };
    });

    if (!data) return res.status(404).json({ error: "Not found" });
    res.json(data);
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getCaseDetail", message: e.message });
  }
}

export async function listCaseMembers(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);

    const members = await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);
      const { rows } = await db.query(
        `
        SELECT
          m.user_id,
          m.role,
          m.is_active,
          m.joined_at,
          u.name as user_name,
          u.email as user_email,
          u.username as user_username
        FROM public.case_members m
        JOIN public.users u ON u.id = m.user_id
        WHERE m.case_id=$1
        ORDER BY m.role, m.joined_at
        `,
        [caseId]
      );
      return rows;
    });

    res.json({ members });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listCaseMembers", message: e.message });
  }
}
// --- helper: user feloldás email/username alapján (opcionális) ---
async function resolveUserId(
  db: any,
  key: { id?: string; email?: string; username?: string }
): Promise<string> {
  if (key.id) return z.string().uuid().parse(key.id);

  if (key.email) {
    const r = await db.query(
      `SELECT id FROM public.users WHERE lower(email)=lower($1) LIMIT 1`,
      [key.email.trim()]
    );
    const rows = (r.rows ?? []) as Array<{ id: string }>;
    if (!rows.length)
      throw new Error(`User nem található email alapján: ${key.email}`);
    return rows[0].id;
  }

  if (key.username) {
    const r = await db.query(
      `SELECT id FROM public.users WHERE lower(username)=lower($1) LIMIT 1`,
      [key.username.trim()]
    );
    const rows = (r.rows ?? []) as Array<{ id: string }>;
    if (!rows.length)
      throw new Error(`User nem található username alapján: ${key.username}`);
    return rows[0].id;
  }

  throw new Error("User azonosító hiányzik (id/email/username).");
}

const AddMembersBody = z
  .object({
    // régi (uuid) mód
    userId: z.string().uuid().optional(),
    userIds: z.array(z.string().uuid()).min(1).optional(),

    // ✅ új (emberi) mód: users: [{email} | {username} | {id}]
    users: z
      .array(
        z.object({
          id: z.string().uuid().optional(),
          email: z.string().email().optional(),
          username: z.string().min(1).optional(),
        })
      )
      .min(1)
      .optional(),

    role: z
      .enum(["owner", "editor", "contributor", "viewer", "external"])
      .default("contributor"),
  })
  .refine((v) => v.userId || v.userIds || v.users, {
    message: "userId vagy userIds vagy users kell.",
  });

export async function addCaseMember(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const b = AddMembersBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      await requireCaseRole(db, caseId, "editor");

      // ✅ összes user id összegyűjtése (uuid + feloldott email/username)
      const ids: string[] = [];

      if (b.userIds?.length) ids.push(...b.userIds);
      if (b.userId) ids.push(b.userId);

      if (b.users?.length) {
        for (const u of b.users) {
          const id = await resolveUserId(db, u);
          ids.push(id);
        }
      }

      // uniq
      const uniq = Array.from(new Set(ids));

      for (const uid of uniq) {
        await db.query(
          `INSERT INTO public.case_members (case_id, user_id, role, is_active)
           VALUES ($1,$2,$3::public.case_role,true)
           ON CONFLICT (case_id, user_id)
           DO UPDATE SET role=EXCLUDED.role, is_active=true`,
          [caseId, uid, b.role]
        );
      }

      void writeAuditLog({
        actorId: req.user?.id ?? null,
        action: "CASE_MEMBER_UPSERT",
        objectType: "case",
        objectId: caseId,
        route: req.originalUrl,
        ip: req.ip,
        payloadHash: null,
        meta: {
          case_ref: caseRef,
          target_user_id: b.userId ?? null,
          target_user_ids: b.userIds ?? null,
          target_users: b.users ?? null,
          role: b.role,
        },
      });
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "addCaseMember", message: e.message });
  }
}

const UpdateCaseBody = z.object({
  title: z.string().min(2).optional(),
  description: z.string().optional(),
  status: z.string().optional(),
  priority: z.number().int().min(1).max(5).optional(),
  dueAt: z.string().datetime().optional(),
  ownerUser: z.string().optional(),
  ownerUserId: z.string().uuid().optional(),
  caseType: z.string().optional(), // ha már van
  flowType: z.enum(["transient", "continuous"]).optional(), // 👈 ADD
  playbookId: z.string().uuid().nullable().optional(), // 👈 ADD (null = törlés)
});
export async function updateCase(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const b = UpdateCaseBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      await requireCaseRole(db, caseId, "editor");

      const sets: string[] = [];
      const params: any[] = [caseId];

      const push = (sql: string, val: any) => {
        params.push(val);
        sets.push(sql.replace("$X", `$${params.length}`));
      };

      if (b.title !== undefined) push(`title=$X`, b.title.trim());
      if (b.description !== undefined)
        push(`description=$X`, b.description ?? null);
      if (b.status !== undefined)
        push(`status=$X::public.case_status`, b.status);
      if (b.priority !== undefined) push(`priority=$X`, b.priority);
      if (b.dueAt !== undefined)
        push(`due_at=$X`, b.dueAt ? new Date(b.dueAt) : null);

      // ✅ NEW: case type (string)
      if (b.caseType !== undefined) {
        push(`case_type=$X`, b.caseType ?? null);
      }

      // ✅ NEW: flow type
      if (b.flowType !== undefined) {
        push(`flow_type=$X`, b.flowType ?? null);
      }

      // ✅ NEW: playbook id (null = törlés)
      if (b.playbookId !== undefined) {
        if (b.playbookId === null) {
          push(`playbook_id=$X`, null);
        } else {
          // validáljuk, hogy létezik-e és aktív-e (A opció playbooknál ez fontos)
          const { rows: pb } = await db.query<{ ok: boolean }>(
            `SELECT EXISTS(
               SELECT 1 FROM public.playbooks
               WHERE id=$1 AND is_active=true
             ) AS ok`,
            [b.playbookId]
          );
          if (!pb[0]?.ok) {
            const e: any = new Error("Playbook not found or inactive");
            e.statusCode = 400;
            throw e;
          }
          push(`playbook_id=$X`, b.playbookId);
        }
      }

      // ✅ owner feloldás: ownerUserId vagy ownerUser (uuid/email/username)
      if (b.ownerUserId !== undefined) {
        push(`owner_user_id=$X`, b.ownerUserId ?? null);
      } else if (b.ownerUser !== undefined) {
        const raw = (b.ownerUser ?? "").trim();

        if (!raw) {
          // üres string esetén töröljük
          push(`owner_user_id=$X`, null);
        } else {
          const asUuid = z.string().uuid().safeParse(raw);
          if (asUuid.success) {
            push(`owner_user_id=$X`, asUuid.data);
          } else if (raw.includes("@")) {
            const ownerId = await resolveUserId(db, { email: raw });
            push(`owner_user_id=$X`, ownerId);
          } else {
            const ownerId = await resolveUserId(db, { username: raw });
            push(`owner_user_id=$X`, ownerId);
          }
        }
      }

      if (!sets.length) return;

      // updated_by + updated_at mindig
      params.push(req.user!.id);
      await db.query(
        `UPDATE public.cases
         SET ${sets.join(", ")},
             updated_by=$${params.length},
             updated_at=now()
         WHERE id=$1`,
        params
      );

      void writeAuditLog({
        actorId: req.user?.id ?? null,
        action: "CASE_UPDATED",
        objectType: "case",
        objectId: caseId,
        route: req.originalUrl,
        ip: req.ip,
        payloadHash: null,
        meta: {
          case_ref: caseRef,
          changed_fields: sets.map((s) => s.split("=")[0]),
        },
      });
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "updateCase", message: e.message });
  }
}

// export async function updateCase(req: Request, res: Response) {
//   try {
//     const caseRef = z.string().min(1).parse(req.params.caseRef);
//     const b = UpdateCaseBody.parse(req.body);

//     await withRlsTx(rlsCtx(req), async (db) => {
//       const caseId = await resolveCaseId(db, caseRef);

//       await requireCaseRole(db, caseId, "editor");

//       const sets: string[] = [];
//       const params: any[] = [caseId];

//       const push = (sql: string, val: any) => {
//         params.push(val);
//         sets.push(sql.replace("$X", `$${params.length}`));
//       };

//       if (b.title !== undefined) push(`title=$X`, b.title.trim());
//       if (b.description !== undefined)
//         push(`description=$X`, b.description ?? null);
//       if (b.status !== undefined)
//         push(`status=$X::public.case_status`, b.status);
//       if (b.priority !== undefined) push(`priority=$X`, b.priority);
//       if (b.dueAt !== undefined)
//         push(`due_at=$X`, b.dueAt ? new Date(b.dueAt) : null);

//       // ✅ case_type (ha nálatok szerkeszthető)
//       if ((b as any).caseType !== undefined) {
//         const v = ((b as any).caseType ?? null) as string | null;
//         push(`case_type=$X`, v);
//       }

//       // ✅ flow_type
//       if ((b as any).flowType !== undefined) {
//         const v = ((b as any).flowType ?? null) as string | null;
//         // DB check constraint úgyis védi, de null-t is engedhetünk, ha akarod:
//         // nálatok a DB DEFAULT + NOT NULL miatt valószínűleg ne adjunk null-t.
//         push(`flow_type=$X`, v ?? "transient");
//       }

//       // ✅ playbook_id (null = törlés)
//       if ((b as any).playbookId !== undefined) {
//         const v = ((b as any).playbookId ?? null) as string | null;
//         push(`playbook_id=$X`, v);
//       }

//       // ✅ owner feloldás: ownerUserId vagy ownerUser (uuid/email/username)
//       if (b.ownerUserId !== undefined) {
//         push(`owner_user_id=$X`, b.ownerUserId ?? null);
//       } else if (b.ownerUser !== undefined) {
//         const raw = b.ownerUser.trim();

//         const asUuid = z.string().uuid().safeParse(raw);
//         if (asUuid.success) {
//           push(`owner_user_id=$X`, asUuid.data);
//         } else if (raw.includes("@")) {
//           const ownerId = await resolveUserId(db, { email: raw });
//           push(`owner_user_id=$X`, ownerId);
//         } else {
//           const ownerId = await resolveUserId(db, { username: raw });
//           push(`owner_user_id=$X`, ownerId);
//         }
//       }

//       if (!sets.length) return;

//       await db.query(
//         `UPDATE public.cases
//          SET ${sets.join(", ")}, updated_at=now()
//          WHERE id=$1`,
//         params
//       );

//       void writeAuditLog({
//         actorId: req.user?.id ?? null,
//         action: "CASE_UPDATED",
//         objectType: "case",
//         objectId: caseId,
//         route: req.originalUrl,
//         ip: req.ip,
//         payloadHash: null,
//         meta: {
//           case_ref: caseRef,
//           changed_fields: sets.map((s) => s.split("=")[0]),
//         },
//       });
//     });

//     res.json({ ok: true });
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "updateCase", message: e.message });
//   }
// }
export async function removeCaseMember(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const userId = z.string().uuid().parse(req.params.userId);

    await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      await requireCaseRole(db, caseId, "editor");

      // Ne lehessen az utolsó owner-t deaktiválni
      const ownersRes = await db.query(
        `
        SELECT count(*)::int AS cnt
        FROM public.case_members
        WHERE case_id=$1
          AND role='owner'::public.case_role
          AND is_active=true
        `,
        [caseId]
      );
      const ownerCount = Number(ownersRes.rows?.[0]?.cnt ?? 0);

      const targetRes = await db.query(
        `
        SELECT role, is_active
        FROM public.case_members
        WHERE case_id=$1 AND user_id=$2
        `,
        [caseId, userId]
      );
      if (!targetRes.rows?.length) {
        const e: any = new Error("Member not found");
        e.statusCode = 404;
        throw e;
      }

      const targetRole = targetRes.rows[0].role as string;
      const isActive = !!targetRes.rows[0].is_active;

      if (isActive && targetRole === "owner" && ownerCount <= 1) {
        const e: any = new Error("Az utolsó owner nem távolítható el.");
        e.statusCode = 400;
        throw e;
      }

      await db.query(
        `
        UPDATE public.case_members
        SET is_active=false
        WHERE case_id=$1 AND user_id=$2
        `,
        [caseId, userId]
      );

      void writeAuditLog({
        actorId: req.user?.id ?? null,
        action: "CASE_MEMBER_REMOVED",
        objectType: "case",
        objectId: caseId,
        route: req.originalUrl,
        ip: req.ip,
        payloadHash: null,
        meta: { case_ref: caseRef, user_id: userId },
      });
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "removeCaseMember", message: e.message });
  }
}
export async function archiveCase(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);

    await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      await requireCaseRole(db, caseId, "editor");

      await db.query(
        `
        UPDATE public.cases
        SET archived_at=now(), archived_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        [caseId, req.user!.id]
      );

      void writeAuditLog({
        actorId: req.user?.id ?? null,
        action: "CASE_ARCHIVED",
        objectType: "case",
        objectId: caseId,
        route: req.originalUrl,
        ip: req.ip,
        payloadHash: null,
        meta: { case_ref: caseRef },
      });
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "archiveCase", message: e.message });
  }
}
const PatchCaseMemberBody = z.object({
  role: z
    .enum(["owner", "editor", "contributor", "viewer", "external"])
    .optional(),
  isActive: z.boolean().optional(),
});

export async function patchCaseMember(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const userId = z.string().uuid().parse(req.params.userId);
    const b = PatchCaseMemberBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      // csak editor tudja
      await requireCaseRole(db, caseId, "editor");

      // last owner védelem (ha owner-t inaktiválod / leváltod)
      if (b.role || b.isActive === false) {
        const targetRes = await db.query(
          `SELECT role, is_active FROM public.case_members WHERE case_id=$1 AND user_id=$2`,
          [caseId, userId]
        );
        if (!targetRes.rows?.length) {
          const e: any = new Error("Member not found");
          e.statusCode = 404;
          throw e;
        }

        const curRole = targetRes.rows[0].role as string;
        const curActive = !!targetRes.rows[0].is_active;

        const willDemoteOwner =
          curRole === "owner" && curActive && b.role && b.role !== "owner";
        const willDeactivateOwner =
          curRole === "owner" && curActive && b.isActive === false;

        if (willDemoteOwner || willDeactivateOwner) {
          const ownersRes = await db.query(
            `SELECT count(*)::int AS cnt
             FROM public.case_members
             WHERE case_id=$1 AND role='owner'::public.case_role AND is_active=true`,
            [caseId]
          );
          const ownerCount = Number(ownersRes.rows?.[0]?.cnt ?? 0);
          if (ownerCount <= 1) {
            const e: any = new Error(
              "Az utolsó owner nem módosítható / nem deaktiválható."
            );
            e.statusCode = 400;
            throw e;
          }
        }
      }

      await db.query(
        `
        UPDATE public.case_members
        SET
          role = COALESCE($3::public.case_role, role),
          is_active = COALESCE($4, is_active)
        WHERE case_id=$1 AND user_id=$2
        `,
        [
          caseId,
          userId,
          b.role ?? null,
          b.isActive === undefined ? null : b.isActive,
        ]
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchCaseMember", message: e.message });
  }
}
export async function deleteCaseMember(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const userId = z.string().uuid().parse(req.params.userId);

    await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);
      await requireCaseRole(db, caseId, "editor");

      // last owner védelem
      const targetRes = await db.query(
        `SELECT role, is_active FROM public.case_members WHERE case_id=$1 AND user_id=$2`,
        [caseId, userId]
      );
      if (!targetRes.rows?.length) {
        const e: any = new Error("Member not found");
        e.statusCode = 404;
        throw e;
      }

      const targetRole = targetRes.rows[0].role as string;
      const isActive = !!targetRes.rows[0].is_active;

      if (isActive && targetRole === "owner") {
        const ownersRes = await db.query(
          `SELECT count(*)::int AS cnt
           FROM public.case_members
           WHERE case_id=$1 AND role='owner'::public.case_role AND is_active=true`,
          [caseId]
        );
        const ownerCount = Number(ownersRes.rows?.[0]?.cnt ?? 0);
        if (ownerCount <= 1) {
          const e: any = new Error("Az utolsó owner nem távolítható el.");
          e.statusCode = 400;
          throw e;
        }
      }

      await db.query(
        `DELETE FROM public.case_members WHERE case_id=$1 AND user_id=$2`,
        [caseId, userId]
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "deleteCaseMember", message: e.message });
  }
}
// Body: we receive final status + ordered list of case ids inside that column
const BoardMoveBody = z.object({
  toStatus: z.string().min(1), // should be public.case_status
  orderedIds: z.array(z.string().uuid()).min(1),
});

export async function moveCaseOnBoard(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const b = BoardMoveBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const caseId = await resolveCaseId(db, caseRef);

      // need editor to reorder/move
      await requireCaseRole(db, caseId, "editor");

      // Safety: ensure moved case is included in orderedIds
      if (!b.orderedIds.includes(caseId)) {
        const e: any = new Error("orderedIds must include the moved case id");
        e.statusCode = 400;
        throw e;
      }

      // We'll set positions as 1000, 2000, 3000... (easy to insert later)
      // numeric lets you later do fractional positions if you want.
      const base = 1000;

      // Update case.status itself (so listCases etc. reflect it)
      await db.query(
        `UPDATE public.cases
         SET status = $2::public.case_status,
             updated_at = now()
         WHERE id = $1`,
        [caseId, b.toStatus]
      );

      // Remove positions for other statuses (optional but keeps table clean)
      await db.query(
        `DELETE FROM public.case_board_positions WHERE case_id = $1 AND status <> $2::public.case_status`,
        [caseId, b.toStatus]
      );

      // Upsert positions for the ENTIRE COLUMN order
      // (only for cases user can see; if not visible, it won't be in orderedIds anyway)
      for (let i = 0; i < b.orderedIds.length; i++) {
        const id = b.orderedIds[i];
        const pos = (i + 1) * base;

        await db.query(
          `
          INSERT INTO public.case_board_positions (case_id, status, position, updated_at)
          VALUES ($1, $2::public.case_status, $3, now())
          ON CONFLICT (case_id, status)
          DO UPDATE SET position = EXCLUDED.position, updated_at = now()
          `,
          [id, b.toStatus, pos]
        );
      }

      void writeAuditLog({
        actorId: req.user?.id ?? null,
        action: "CASE_BOARD_MOVED",
        objectType: "case",
        objectId: caseId,
        route: req.originalUrl,
        ip: req.ip,
        payloadHash: null,
        meta: {
          case_ref: caseRef,
          to_status: b.toStatus,
          count_in_column: b.orderedIds.length,
        },
      });
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "moveCaseOnBoard", message: e.message });
  }
}
export async function unarchiveCase(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);

    await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      // same permission as archive
      await requireCaseRole(db, caseId, "editor");

      await db.query(
        `
        UPDATE public.cases
        SET archived_at=NULL, archived_by=NULL, updated_at=now()
        WHERE id=$1 AND archived_at IS NOT NULL
        `,
        [caseId]
      );

      void writeAuditLog({
        actorId: req.user?.id ?? null,
        action: "CASE_UNARCHIVED",
        objectType: "case",
        objectId: caseId,
        route: req.originalUrl,
        ip: req.ip,
        payloadHash: null,
        meta: { case_ref: caseRef },
      });
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "unarchiveCase", message: e.message });
  }
}
