import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";
import { requireCaseRole } from "../../../../utils/aclCase.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";

const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

function truthy(v: unknown, defaultValue = false) {
  if (v === undefined || v === null) return defaultValue;
  if (typeof v === "boolean") return v;
  if (typeof v === "string") return v === "1" || v === "true";
  return defaultValue;
}

async function resolveCaseId(db: Db, caseRef: string): Promise<string> {
  const ref = String(caseRef ?? "").trim();
  if (UUID_RE.test(ref)) return z.string().uuid().parse(ref);

  const r = await db.query(
    `SELECT id FROM public.cases WHERE code=$1 LIMIT 1`,
    [ref]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Case not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id;
}

async function resolvePlaybookId(db: Db, playbookRef: string): Promise<string> {
  const ref = String(playbookRef ?? "").trim();
  if (!ref) {
    const e: any = new Error("playbookRef is required");
    e.statusCode = 400;
    throw e;
  }

  if (UUID_RE.test(ref)) return z.string().uuid().parse(ref);

  // key alapján
  const r = await db.query(
    `SELECT id FROM public.playbooks WHERE key=$1 LIMIT 1`,
    [ref]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Playbook not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id;
}

function buildKickoffNotes(opts: {
  playbookName: string;
  playbookKey: string;
  playbookDescription: string | null;
  steps: Array<{
    sort_order: number;
    title: string;
    description: string | null;
  }>;
}) {
  const lines: string[] = [];
  lines.push(`# Kickoff`);
  lines.push(`Playbook: **${opts.playbookName}** (\`${opts.playbookKey}\`)`);
  if (opts.playbookDescription?.trim()) {
    lines.push("");
    lines.push(opts.playbookDescription.trim());
  }

  lines.push("");
  lines.push("## Checklist / Steps");
  for (const s of opts.steps) {
    lines.push(`- [ ] ${s.title}${s.description ? ` — ${s.description}` : ""}`);
  }
  lines.push("");

  return lines.join("\n");
}

// -----------------------------------------------------
// BODY
// -----------------------------------------------------
const ApplyPlaybookBody = z.object({
  playbookRef: z.string().min(1),

  // ha true: case_type/flow_type mehet playbook defaultjára akkor is, ha volt már
  overwrite: z
    .union([z.literal("1"), z.literal("true"), z.boolean()])
    .optional(),

  // checklist state rows létrehozása
  createChecklist: z
    .union([z.literal("1"), z.literal("true"), z.boolean()])
    .optional(),

  // ha true: ugyanazon playbook re-apply esetén újragenerálja a checklist state-et (reset)
  overwriteChecklist: z
    .union([z.literal("1"), z.literal("true"), z.boolean()])
    .optional(),

  // kickoff meeting (opcionális)
  createKickoffMeeting: z
    .union([z.literal("1"), z.literal("true"), z.boolean()])
    .optional(),

  kickoffTitle: z.string().optional(),
  kickoffAgendaTitle: z.string().optional(),
  kickoffStartsAt: z.string().optional(),
  kickoffEndsAt: z.string().optional(),
});

// -----------------------------------------------------
// 1) APPLY PLAYBOOK TO CASE
// -----------------------------------------------------
export async function applyPlaybookToCase(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const b = ApplyPlaybookBody.parse(req.body);

    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const caseId = await resolveCaseId(db, caseRef);

      // jogosultság: legalább editor a case-en
      await requireCaseRole(db, caseId, "editor");

      const playbookId = await resolvePlaybookId(db, b.playbookRef);

      const { rows: pbRows } = await db.query<{
        id: string;
        key: string;
        name: string;
        description: string | null;
        default_case_type: string | null;
        default_flow_type: string | null;
        is_active: boolean;
      }>(
        `
        SELECT
          id, key, name, description,
          default_case_type, default_flow_type,
          is_active
        FROM public.playbooks
        WHERE id=$1
        LIMIT 1
        `,
        [playbookId]
      );

      if (!pbRows.length) {
        const e: any = new Error("Playbook not found");
        e.statusCode = 404;
        throw e;
      }
      if (!pbRows[0].is_active) {
        const e: any = new Error("Playbook is inactive");
        e.statusCode = 400;
        throw e;
      }
      const pb = pbRows[0];

      const { rows: caseRows } = await db.query<{
        id: string;
        title: string;
        case_type: string | null;
        flow_type: string | null;
        playbook_id: string | null;
      }>(
        `
        SELECT id, title, case_type, flow_type, playbook_id
        FROM public.cases
        WHERE id=$1
        LIMIT 1
        `,
        [caseId]
      );

      if (!caseRows.length) {
        const e: any = new Error("Case not found");
        e.statusCode = 404;
        throw e;
      }
      const c = caseRows[0];

      // playbook steps
      const { rows: steps } = await db.query<{
        id: string;
        sort_order: number;
        title: string;
        description: string | null;
      }>(
        `
        SELECT id, sort_order, title, description
        FROM public.playbook_steps
        WHERE playbook_id=$1
        ORDER BY sort_order ASC, title ASC
        `,
        [playbookId]
      );

      const overwrite = truthy(b.overwrite, false);

      const nextCaseType = overwrite
        ? pb.default_case_type ?? c.case_type
        : c.case_type?.trim()
        ? c.case_type
        : pb.default_case_type ?? c.case_type;

      const nextFlowType = overwrite
        ? pb.default_flow_type ?? c.flow_type
        : c.flow_type?.trim()
        ? c.flow_type
        : pb.default_flow_type ?? c.flow_type;

      // update case fields
      await db.query(
        `
        UPDATE public.cases
        SET
          playbook_id = $2,
          case_type   = $3,
          flow_type   = $4,
          updated_by  = $5::uuid,
          updated_at  = now()
        WHERE id=$1
        `,
        [
          caseId,
          playbookId,
          nextCaseType ?? null,
          nextFlowType ?? null,
          req.user!.id,
        ]
      );

      // -------------------------------------------
      // Checklist state rows (case_playbook_step_states)
      // schema szerint: (case_id, step_id, is_done, done_at, done_by, notes, ...)
      // -------------------------------------------
      const createChecklist = truthy(b.createChecklist, true);
      const overwriteChecklist = truthy(b.overwriteChecklist, false);

      // ha playbook váltás történt: töröljük a régi step state-eket (különben keveredik)
      //   const playbookChanged =
      //     (c.playbook_id ?? null) && (c.playbook_id ?? null) !== playbookId;
      const playbookChanged = !!c.playbook_id && c.playbook_id !== playbookId;

      if (createChecklist) {
        if (playbookChanged || overwriteChecklist) {
          await db.query(
            `DELETE FROM public.case_playbook_step_states WHERE case_id=$1`,
            [caseId]
          );
        }

        // insert all steps (missing only), keep existing states unless we deleted
        for (const s of steps) {
          await db.query(
            `
            INSERT INTO public.case_playbook_step_states
              (case_id, step_id, is_done, done_at, done_by, notes, created_at, updated_at)
            VALUES
              ($1, $2, false, NULL, NULL, NULL, now(), now())
            ON CONFLICT (case_id, step_id)
            DO NOTHING
            `,
            [caseId, s.id]
          );
        }
      }

      // -------------------------------------------
      // Kickoff meeting (opcionális)
      // -------------------------------------------
      let meetingId: string | null = null;

      const createKickoffMeeting = truthy(b.createKickoffMeeting, false);

      if (createKickoffMeeting) {
        const startsAt = b.kickoffStartsAt
          ? new Date(b.kickoffStartsAt)
          : new Date();
        const endsAt = b.kickoffEndsAt ? new Date(b.kickoffEndsAt) : null;

        const kickoffTitle =
          b.kickoffTitle?.trim() || `Kickoff – ${c.title?.trim() || "Case"}`;

        const notes = buildKickoffNotes({
          playbookName: pb.name,
          playbookKey: pb.key,
          playbookDescription: pb.description,
          steps: steps.map((x) => ({
            sort_order: Number(x.sort_order ?? 0),
            title: x.title,
            description: x.description ?? null,
          })),
        });

        const { rows: m } = await db.query<{ id: string; code: string }>(
          `
          INSERT INTO public.meetings (id, title, starts_at, ends_at, notes, created_by)
          VALUES (gen_random_uuid(), $1, $2, $3, $4, $5)
          RETURNING id, code
          `,
          [kickoffTitle, startsAt, endsAt, notes, req.user!.id]
        );

        meetingId = m[0]?.id ?? null;

        if (meetingId) {
          const agendaTitle = b.kickoffAgendaTitle?.trim() || "Kickoff";

          await db.query(
            `
            INSERT INTO public.meeting_agenda_items
              (id, meeting_id, case_id, thread_id, title, sort_order)
            VALUES
              (gen_random_uuid(), $1, $2, NULL, $3, 0)
            `,
            [meetingId, caseId, agendaTitle]
          );
        }
      }

      void writeAuditLog({
        actorId: req.user?.id ?? null,
        action: "CASE_PLAYBOOK_APPLIED",
        objectType: "case",
        objectId: caseId,
        route: req.originalUrl,
        ip: req.ip,
        payloadHash: null,
        meta: {
          case_ref: caseRef,
          playbook_ref: b.playbookRef,
          playbook_id: playbookId,
          overwrite,
          create_checklist: createChecklist,
          overwrite_checklist: overwriteChecklist,
          changed: {
            playbook_id: playbookId,
            case_type: nextCaseType ?? null,
            flow_type: nextFlowType ?? null,
          },
          kickoff_meeting_id: meetingId,
        },
      });

      return { ok: true, caseId, playbookId, meetingId };
    });

    res.status(200).json(out);
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "applyPlaybookToCase", message: e.message });
  }
}

// -----------------------------------------------------
// 2) GET CASE PLAYBOOK CHECKLIST
// -----------------------------------------------------
export async function getCasePlaybookChecklist(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);

    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const caseId = await resolveCaseId(db, caseRef);

      // read jog
      await requireCaseRole(db, caseId, "viewer");

      const { rows: cRows } = await db.query<{ playbook_id: string | null }>(
        `SELECT playbook_id FROM public.cases WHERE id=$1 LIMIT 1`,
        [caseId]
      );
      const playbookId = cRows[0]?.playbook_id ?? null;

      if (!playbookId) return { caseId, playbookId: null, steps: [] };

      const { rows } = await db.query<{
        step_id: string;
        sort_order: number;
        title: string;
        description: string | null;
        is_done: boolean;
        done_at: string | null;
        done_by: string | null;
        notes: string | null;
      }>(
        `
        SELECT
          s.id as step_id,
          s.sort_order,
          s.title,
          s.description,
          COALESCE(st.is_done, false) as is_done,
          st.done_at,
          st.done_by,
          st.notes
        FROM public.playbook_steps s
        LEFT JOIN public.case_playbook_step_states st
          ON st.step_id = s.id
         AND st.case_id = $1
        WHERE s.playbook_id = $2
        ORDER BY s.sort_order ASC, s.title ASC
        `,
        [caseId, playbookId]
      );

      return { caseId, playbookId, steps: rows };
    });

    res.json(out);
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getCasePlaybookChecklist", message: e.message });
  }
}

// -----------------------------------------------------
// 3) PATCH CHECKLIST STEP (done/undone + notes)
// -----------------------------------------------------
const PatchChecklistBody = z.object({
  isDone: z.boolean(),
  notes: z.string().optional().nullable(),
});

export async function patchCasePlaybookChecklistStep(
  req: Request,
  res: Response
) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const stepRef = z.string().min(1).parse(req.params.stepRef);
    const actorId = z.string().uuid().parse(req.user!.id);

    const b = PatchChecklistBody.parse(req.body);

    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const caseId = await resolveCaseId(db, caseRef);

      await requireCaseRole(db, caseId, "editor");

      // stepRef: playbook_steps.id (uuid)
      const stepId = z.string().uuid().parse(stepRef);

      // ✅ actor id: castoljuk uuid-ra (query-ben is)
      //   const actorId = req.user!.id; // ha biztos uuid string, ok így

      const { rows } = await db.query<{
        id: string;
        case_id: string;
        step_id: string;
        is_done: boolean;
        done_at: string | null;
        done_by: string | null;
        notes: string | null;
        updated_at: string;
      }>(
        `
        INSERT INTO public.case_playbook_step_states
          (case_id, step_id, is_done, done_at, done_by, notes, created_at, updated_at)
        VALUES
          (
            $1,
            $2,
            $3,
            CASE WHEN $3 THEN now() ELSE NULL END,
            CASE WHEN $3 THEN $4::uuid ELSE NULL END,
            $5,
            now(),
            now()
          )
        ON CONFLICT (case_id, step_id)
        DO UPDATE SET
          is_done   = EXCLUDED.is_done,
          done_at   = EXCLUDED.done_at,
          done_by   = EXCLUDED.done_by,
          notes     = EXCLUDED.notes,
          updated_at = now()
        RETURNING id, case_id, step_id, is_done, done_at, done_by, notes, updated_at
        `,
        [caseId, stepId, b.isDone, actorId, b.notes ?? null]
      );

      return rows[0];
    });

    res.json({ ok: true, state: out });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchCasePlaybookChecklistStep", message: e.message });
  }
}

// export async function patchCasePlaybookChecklistStep(
//   req: Request,
//   res: Response
// ) {
//   try {
//     const caseRef = z.string().min(1).parse(req.params.caseRef);
//     const stepRef = z.string().min(1).parse(req.params.stepRef);
//     const b = PatchChecklistBody.parse(req.body);

//     const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
//       const caseId = await resolveCaseId(db, caseRef);

//       await requireCaseRole(db, caseId, "editor");

//       // stepRef: playbook_steps.id (uuid)
//       const stepId = z.string().uuid().parse(stepRef);

//       const { rows } = await db.query<{
//         id: string;
//         case_id: string;
//         step_id: string;
//         is_done: boolean;
//         done_at: string | null;
//         done_by: string | null;
//         notes: string | null;
//         updated_at: string;
//       }>(
//         `
//         INSERT INTO public.case_playbook_step_states
//           (case_id, step_id, is_done, done_at, done_by, notes, created_at, updated_at)
//         VALUES
//           ($1, $2, $3, CASE WHEN $3 THEN now() ELSE NULL END, CASE WHEN $3 THEN $4 ELSE NULL END, $5, now(), now())
//         ON CONFLICT (case_id, step_id)
//         DO UPDATE SET
//           is_done = EXCLUDED.is_done,
//           done_at = EXCLUDED.done_at,
//           done_by = EXCLUDED.done_by,
//           notes   = EXCLUDED.notes,
//           updated_at = now()
//         RETURNING id, case_id, step_id, is_done, done_at, done_by, notes, updated_at
//         `,
//         [caseId, stepId, b.isDone, req.user!.id, b.notes ?? null]
//       );

//       return rows[0];
//     });

//     res.json({ ok: true, state: out });
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "patchCasePlaybookChecklistStep", message: e.message });
//   }
// }

// import type { Request, Response } from "express";
// import { z } from "zod";
// import { withRlsTx, Db } from "@/lib/postgres.js";
// import { requireCaseRole } from "../../../../utils/aclCase.js";
// import { writeAuditLog } from "../../../../audit/writeAuditLog.js";

// const UUID_RE =
//   /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

// function rlsCtx(req: Request) {
//   return {
//     userId: req.user!.id,
//     hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
//   };
// }

// async function resolveCaseId(db: Db, caseRef: string): Promise<string> {
//   const ref = String(caseRef ?? "").trim();
//   if (UUID_RE.test(ref)) return z.string().uuid().parse(ref);

//   const r = await db.query(
//     `SELECT id FROM public.cases WHERE code=$1 LIMIT 1`,
//     [ref]
//   );
//   if (!r.rows?.length) {
//     const e: any = new Error("Case not found");
//     e.statusCode = 404;
//     throw e;
//   }
//   return r.rows[0].id;
// }

// async function resolvePlaybookId(db: Db, playbookRef: string): Promise<string> {
//   const ref = String(playbookRef ?? "").trim();
//   if (!ref) {
//     const e: any = new Error("playbookRef is required");
//     e.statusCode = 400;
//     throw e;
//   }

//   if (UUID_RE.test(ref)) return z.string().uuid().parse(ref);

//   // key alapján
//   const r = await db.query(
//     `SELECT id FROM public.playbooks WHERE key=$1 LIMIT 1`,
//     [ref]
//   );
//   if (!r.rows?.length) {
//     const e: any = new Error("Playbook not found");
//     e.statusCode = 404;
//     throw e;
//   }
//   return r.rows[0].id;
// }

// function buildKickoffNotes(opts: {
//   playbookName: string;
//   playbookKey: string;
//   playbookDescription: string | null;
//   steps: Array<{
//     sort_order: number;
//     title: string;
//     description: string | null;
//   }>;
// }) {
//   const lines: string[] = [];
//   lines.push(`# Kickoff`);
//   lines.push(`Playbook: **${opts.playbookName}** (\`${opts.playbookKey}\`)`);
//   if (opts.playbookDescription?.trim()) {
//     lines.push("");
//     lines.push(opts.playbookDescription.trim());
//   }

//   lines.push("");
//   lines.push("## Checklist / Steps");
//   for (const s of opts.steps) {
//     lines.push(`- [ ] ${s.title}${s.description ? ` — ${s.description}` : ""}`);
//   }
//   lines.push("");

//   return lines.join("\n");
// }
// const ApplyPlaybookBody = z.object({
//   playbookRef: z.string().min(1),
//   overwrite: z
//     .union([z.literal("1"), z.literal("true"), z.boolean()])
//     .optional(),

//   // checklist state rows létrehozása (ha kimarad: default true)
//   createChecklist: z
//     .union([z.literal("1"), z.literal("true"), z.boolean()])
//     .optional(),
//   overwriteChecklist: z.coerce.boolean().optional().default(false),
//   // opcionális kickoff meeting generálás (ha van nálatok meetings modul)
//   createKickoffMeeting: z
//     .union([z.literal("1"), z.literal("true"), z.boolean()])
//     .optional(),

//   kickoffTitle: z.string().optional(),
//   kickoffAgendaTitle: z.string().optional(),
//   kickoffStartsAt: z.string().optional(),
//   kickoffEndsAt: z.string().optional(),
// });
// // const ApplyPlaybookBody = z.object({
// //   playbookRef: z.string().min(1),

// //   // ha true: case_type/flow_type is mehet a playbook defaultjára akkor is, ha volt már
// //   overwrite: z.coerce.boolean().optional().default(false),

// //   // checklist generálás playbook step-ekből (action_items)
// //   createChecklist: z.coerce.boolean().optional().default(true),

// //   // ha true: törli az adott case-hez tartozó korábbi playbook-step action itemeket, és újragenerál
// //   overwriteChecklist: z.coerce.boolean().optional().default(false),

// //   // kickoff meeting opciók
// //   createKickoffMeeting: z.coerce.boolean().optional().default(false),
// //   kickoffStartsAt: z.string().optional(),
// //   kickoffEndsAt: z.string().optional(),
// //   kickoffTitle: z.string().optional(),
// //   kickoffAgendaTitle: z.string().optional(),
// // });

// // -----------------------------------------------------
// // 1) APPLY PLAYBOOK TO CASE
// //   - case.playbook_id beállítás
// //   - default case_type / flow_type (overwrite vagy “csak ha üres”)
// //   - opcionálisan checklist state row-k a case_playbook_step_states táblába
// //   - opcionálisan kickoff meeting + agenda item
// // -----------------------------------------------------
// export async function applyPlaybookToCase(req: Request, res: Response) {
//   try {
//     const caseRef = z.string().min(1).parse(req.params.caseRef);
//     const b = ApplyPlaybookBody.parse(req.body);

//     const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
//       const caseId = await resolveCaseId(db, caseRef);

//       // jogosultság: legalább editor a case-en
//       await requireCaseRole(db, caseId, "editor");

//       const playbookId = await resolvePlaybookId(db, b.playbookRef);

//       const { rows: pbRows } = await db.query<{
//         id: string;
//         key: string;
//         name: string;
//         description: string | null;
//         default_case_type: string | null;
//         default_flow_type: string | null;
//         is_active: boolean;
//       }>(
//         `
//         SELECT
//           id, key, name, description,
//           default_case_type, default_flow_type,
//           is_active
//         FROM public.playbooks
//         WHERE id=$1
//         LIMIT 1
//         `,
//         [playbookId]
//       );

//       if (!pbRows.length) {
//         const e: any = new Error("Playbook not found");
//         e.statusCode = 404;
//         throw e;
//       }
//       if (!pbRows[0].is_active) {
//         const e: any = new Error("Playbook is inactive");
//         e.statusCode = 400;
//         throw e;
//       }
//       const pb = pbRows[0];

//       const { rows: caseRows } = await db.query<{
//         id: string;
//         title: string;
//         case_type: string | null;
//         flow_type: string | null;
//         playbook_id: string | null;
//       }>(
//         `
//         SELECT id, title, case_type, flow_type, playbook_id
//         FROM public.cases
//         WHERE id=$1
//         LIMIT 1
//         `,
//         [caseId]
//       );

//       if (!caseRows.length) {
//         const e: any = new Error("Case not found");
//         e.statusCode = 404;
//         throw e;
//       }
//       const c = caseRows[0];

//       // playbook steps
//       const { rows: steps } = await db.query<{
//         id: string;
//         sort_order: number;
//         title: string;
//         description: string | null;
//       }>(
//         `
//         SELECT id, sort_order, title, description
//         FROM public.playbook_steps
//         WHERE playbook_id=$1
//         ORDER BY sort_order ASC, title ASC
//         `,
//         [playbookId]
//       );

//       const overwrite =
//         b.overwrite === true || b.overwrite === "1" || b.overwrite === "true";

//       const nextCaseType = overwrite
//         ? pb.default_case_type ?? c.case_type
//         : c.case_type?.trim()
//         ? c.case_type
//         : pb.default_case_type ?? c.case_type;

//       const nextFlowType = overwrite
//         ? pb.default_flow_type ?? c.flow_type
//         : c.flow_type?.trim()
//         ? c.flow_type
//         : pb.default_flow_type ?? c.flow_type;

//       await db.query(
//         `
//         UPDATE public.cases
//         SET
//           playbook_id = $2,
//           case_type   = $3,
//           flow_type   = $4,
//           updated_by  = $5,
//           updated_at  = now()
//         WHERE id=$1
//         `,
//         [
//           caseId,
//           playbookId,
//           nextCaseType ?? null,
//           nextFlowType ?? null,
//           req.user!.id,
//         ]
//       );

//       // -------------------------------------------
//       // Checklist state rows
//       // -------------------------------------------
//       const createChecklist =
//         b.createChecklist === undefined
//           ? true
//           : b.createChecklist === true ||
//             b.createChecklist === "1" ||
//             b.createChecklist === "true";

//       if (createChecklist) {
//         // Biztonság: a playbook váltásnál a korábbi állapotok törlése
//         await db.query(
//           `DELETE FROM public.case_playbook_step_states WHERE case_id=$1`,
//           [caseId]
//         );

//         for (const s of steps) {
//           await db.query(
//             `
//             INSERT INTO public.case_playbook_step_states
//               (case_id, playbook_step_id, is_done, done_at, done_by, created_at, updated_at)
//             VALUES
//               ($1, $2, false, NULL, NULL, now(), now())
//             `,
//             [caseId, s.id]
//           );
//         }
//       }

//       // -------------------------------------------
//       // Kickoff meeting (opcionális)
//       // -------------------------------------------
//       let meetingId: string | null = null;

//       const createKickoffMeeting =
//         b.createKickoffMeeting === true ||
//         b.createKickoffMeeting === "1" ||
//         b.createKickoffMeeting === "true";

//       if (createKickoffMeeting) {
//         const startsAt = b.kickoffStartsAt
//           ? new Date(b.kickoffStartsAt)
//           : new Date();
//         const endsAt = b.kickoffEndsAt ? new Date(b.kickoffEndsAt) : null;

//         const kickoffTitle =
//           b.kickoffTitle?.trim() || `Kickoff – ${c.title?.trim() || "Case"}`;

//         const notes = buildKickoffNotes({
//           playbookName: pb.name,
//           playbookKey: pb.key,
//           playbookDescription: pb.description,
//           steps: steps.map((x) => ({
//             sort_order: Number(x.sort_order ?? 0),
//             title: x.title,
//             description: x.description ?? null,
//           })),
//         });

//         const { rows: m } = await db.query<{ id: string; code: string }>(
//           `
//           INSERT INTO public.meetings (id, title, starts_at, ends_at, notes, created_by)
//           VALUES (gen_random_uuid(), $1, $2, $3, $4, $5)
//           RETURNING id, code
//           `,
//           [kickoffTitle, startsAt, endsAt, notes, req.user!.id]
//         );

//         meetingId = m[0]?.id ?? null;

//         if (meetingId) {
//           const agendaTitle = b.kickoffAgendaTitle?.trim() || "Kickoff";

//           await db.query(
//             `
//             INSERT INTO public.meeting_agenda_items
//               (id, meeting_id, case_id, thread_id, title, sort_order)
//             VALUES
//               (gen_random_uuid(), $1, $2, NULL, $3, 0)
//             `,
//             [meetingId, caseId, agendaTitle]
//           );
//         }
//       }

//       void writeAuditLog({
//         actorId: req.user?.id ?? null,
//         action: "CASE_PLAYBOOK_APPLIED",
//         objectType: "case",
//         objectId: caseId,
//         route: req.originalUrl,
//         ip: req.ip,
//         payloadHash: null,
//         meta: {
//           case_ref: caseRef,
//           playbook_ref: b.playbookRef,
//           playbook_id: playbookId,
//           overwrite,
//           create_checklist: createChecklist,
//           changed: {
//             playbook_id: playbookId,
//             case_type: nextCaseType ?? null,
//             flow_type: nextFlowType ?? null,
//           },
//           kickoff_meeting_id: meetingId,
//         },
//       });

//       return { ok: true, caseId, playbookId, meetingId };
//     });

//     res.status(200).json(out);
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "applyPlaybookToCase", message: e.message });
//   }
// }

// // -----------------------------------------------------
// // 2) GET CASE PLAYBOOK CHECKLIST
// // -----------------------------------------------------
// export async function getCasePlaybookChecklist(req: Request, res: Response) {
//   try {
//     const caseRef = z.string().min(1).parse(req.params.caseRef);

//     const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
//       const caseId = await resolveCaseId(db, caseRef);

//       // read jog
//       await requireCaseRole(db, caseId, "viewer");

//       // case -> playbook
//       const { rows: cRows } = await db.query<{ playbook_id: string | null }>(
//         `SELECT playbook_id FROM public.cases WHERE id=$1 LIMIT 1`,
//         [caseId]
//       );
//       const playbookId = cRows[0]?.playbook_id ?? null;

//       if (!playbookId) {
//         return { caseId, playbookId: null, steps: [] };
//       }

//       const { rows } = await db.query<{
//         step_id: string;
//         sort_order: number;
//         title: string;
//         description: string | null;
//         is_done: boolean;
//         done_at: string | null;
//         done_by: string | null;
//       }>(
//         `
//         SELECT
//           s.id as step_id,
//           s.sort_order,
//           s.title,
//           s.description,
//           COALESCE(st.is_done, false) as is_done,
//           st.done_at,
//           st.done_by
//         FROM public.playbook_steps s
//         LEFT JOIN public.case_playbook_step_states st
//           ON st.playbook_step_id = s.id
//          AND st.case_id = $1
//         WHERE s.playbook_id = $2
//         ORDER BY s.sort_order ASC, s.title ASC
//         `,
//         [caseId, playbookId]
//       );

//       return {
//         caseId,
//         playbookId,
//         steps: rows,
//       };
//     });

//     res.json(out);
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "getCasePlaybookChecklist", message: e.message });
//   }
// }

// // -----------------------------------------------------
// // 3) SET CHECKLIST STEP STATE (done/undone)
// // -----------------------------------------------------
// const SetStepStateBody = z.object({
//   isDone: z.boolean(),
// });

// export async function setCasePlaybookStepState(req: Request, res: Response) {
//   try {
//     const caseRef = z.string().min(1).parse(req.params.caseRef);
//     const stepId = z.string().uuid().parse(req.params.stepId);
//     const b = SetStepStateBody.parse(req.body);

//     const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
//       const caseId = await resolveCaseId(db, caseRef);

//       await requireCaseRole(db, caseId, "editor");

//       if (b.isDone) {
//         await db.query(
//           `
//           INSERT INTO public.case_playbook_step_states
//             (case_id, playbook_step_id, is_done, done_at, done_by, created_at, updated_at)
//           VALUES
//             ($1, $2, true, now(), $3, now(), now())
//           ON CONFLICT (case_id, playbook_step_id)
//           DO UPDATE SET
//             is_done = true,
//             done_at = now(),
//             done_by = EXCLUDED.done_by,
//             updated_at = now()
//           `,
//           [caseId, stepId, req.user!.id]
//         );
//       } else {
//         await db.query(
//           `
//           INSERT INTO public.case_playbook_step_states
//             (case_id, playbook_step_id, is_done, done_at, done_by, created_at, updated_at)
//           VALUES
//             ($1, $2, false, NULL, NULL, now(), now())
//           ON CONFLICT (case_id, playbook_step_id)
//           DO UPDATE SET
//             is_done = false,
//             done_at = NULL,
//             done_by = NULL,
//             updated_at = now()
//           `,
//           [caseId, stepId]
//         );
//       }

//       return { ok: true };
//     });

//     res.json(out);
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "setCasePlaybookStepState", message: e.message });
//   }
// }

// const PatchChecklistBody = z.object({
//   isDone: z.boolean(),
//   notes: z.string().optional().nullable(),
// });

// // PATCH /api/app/cases/:caseRef/playbook-checklist/:stepRef
// export async function patchCasePlaybookChecklistStep(
//   req: Request,
//   res: Response
// ) {
//   try {
//     const caseRef = z.string().min(1).parse(req.params.caseRef);
//     const stepRef = z.string().min(1).parse(req.params.stepRef);
//     const b = PatchChecklistBody.parse(req.body);

//     const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
//       const caseId = await resolveCaseId(db, caseRef);

//       // stepRef is expected to be UUID of playbook_steps.id
//       const stepId = z.string().uuid().parse(stepRef);

//       const doneAt = b.isDone ? new Date().toISOString() : null;
//       const doneBy = b.isDone ? req.user!.id : null;

//       const { rows } = await db.query(
//         `
//         INSERT INTO public.case_playbook_step_states
//           (case_id, step_id, is_done, done_at, done_by, notes)
//         VALUES
//           ($1, $2, $3, $4, $5, $6)
//         ON CONFLICT (case_id, step_id)
//         DO UPDATE SET
//           is_done = EXCLUDED.is_done,
//           done_at = EXCLUDED.done_at,
//           done_by = EXCLUDED.done_by,
//           notes = EXCLUDED.notes
//         RETURNING id, case_id, step_id, is_done, done_at, done_by, notes, updated_at
//         `,
//         [caseId, stepId, b.isDone, doneAt, doneBy, b.notes ?? null]
//       );

//       return rows[0];
//     });

//     res.json({ ok: true, state: out });
//   } catch (e: any) {
//     res
//       .status(e.statusCode ?? 400)
//       .json({ error: "patchCasePlaybookChecklistStep", message: e.message });
//   }
// }
