import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx } from "@/lib/postgres.js";

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

const LookupQuery = z.object({
  q: z.string().optional(),
  limit: z.coerce.number().int().min(1).max(50).optional(),
  caseId: z.string().uuid().optional(), // threads szűréshez
});

export async function lookupUsers(req: Request, res: Response) {
  try {
    const q = LookupQuery.parse(req.query);
    const limit = q.limit ?? 20;
    const term = (q.q ?? "").trim();

    const users = await withRlsTx(rlsCtx(req), async (db) => {
      const params: any[] = [limit];
      let where = `u.is_active=true`;

      if (term) {
        params.push(`%${term}%`);
        where += ` AND (u.name ILIKE $2 OR u.email ILIKE $2 OR u.username ILIKE $2)`;
      }

      const { rows } = await db.query(
        `
        SELECT u.id, u.name, u.email, u.username
        FROM public.users u
        WHERE ${where}
        ORDER BY u.name ASC
        LIMIT $1
        `,
        params
      );

      return rows;
    });

    res.json({ users });
  } catch (e: any) {
    res.status(400).json({ error: "lookupUsers", message: e.message });
  }
}

export async function lookupCases(req: Request, res: Response) {
  try {
    const q = LookupQuery.parse(req.query);
    const limit = q.limit ?? 20;
    const term = (q.q ?? "").trim();

    const cases = await withRlsTx(rlsCtx(req), async (db) => {
      const params: any[] = [limit];
      let where = `TRUE`; // RLS úgyis szűr

      if (term) {
        params.push(`%${term}%`);
        where += ` AND (c.title ILIKE $2 OR COALESCE(c.description,'') ILIKE $2)`;
      }

      const { rows } = await db.query(
        `
        SELECT c.id, c.title, c.status, c.priority, c.due_at, c.updated_at
        FROM public.cases c
        WHERE ${where}
        ORDER BY c.updated_at DESC
        LIMIT $1
        `,
        params
      );

      return rows;
    });

    res.json({ cases });
  } catch (e: any) {
    res.status(400).json({ error: "lookupCases", message: e.message });
  }
}

export async function lookupThreads(req: Request, res: Response) {
  try {
    const q = LookupQuery.parse(req.query);
    const limit = q.limit ?? 20;
    const term = (q.q ?? "").trim();

    const threads = await withRlsTx(rlsCtx(req), async (db) => {
      const params: any[] = [limit];
      const where: string[] = [];

      if (q.caseId) {
        params.push(q.caseId);
        where.push(`t.case_id = $${params.length}::uuid`);
      }

      if (term) {
        params.push(`%${term}%`);
        where.push(
          `(t.title ILIKE $${params.length} OR COALESCE(t.description,'') ILIKE $${params.length})`
        );
      }

      const { rows } = await db.query(
        `
        SELECT
          t.id, t.case_id, t.title, t.input_due_at, t.updated_at,
          c.title as case_title
        FROM public.case_threads t
        JOIN public.cases c ON c.id=t.case_id
        ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
        ORDER BY t.updated_at DESC
        LIMIT $1
        `,
        params
      );

      return rows;
    });

    res.json({ threads });
  } catch (e: any) {
    res.status(400).json({ error: "lookupThreads", message: e.message });
  }
}
// A-opcióhoz: egyszerű dropdownhoz elég a distinct lista.
// (Később lehet “hivatalos” lista/config, de indulásnak ez jó és nem merev.)
export async function listCaseTypes(req: Request, res: Response) {
  try {
    const types = await withRlsTx(rlsCtx(req), async (db) => {
      const r = await db.query<{ case_type: string | null }>(
        `
        SELECT DISTINCT c.case_type
        FROM public.cases c
        WHERE c.case_type IS NOT NULL AND btrim(c.case_type) <> ''
        ORDER BY c.case_type ASC
        `
      );
      return r.rows.map((x) => x.case_type!).filter(Boolean);
    });

    res.json({ caseTypes: types });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listCaseTypes", message: e.message });
  }
}
