// src/app/controllers/api/admin/me/me.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

export async function getMe(req: Request, res: Response) {
  try {
    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const userId = req.user!.id;

      const { rows: pref } = await db.query<{ dashboard_cases: any }>(
        `SELECT dashboard_cases FROM public.user_preferences WHERE user_id=$1 LIMIT 1`,
        [userId]
      );

      const dashboardCases = pref[0]?.dashboard_cases ?? {};

      const { rows: u } = await db.query<{
        id: string;
        name: string;
        email: string | null;
        username: string;
      }>(
        `
  SELECT id, name, email, username
  FROM public.users
  WHERE id=$1
  LIMIT 1
  `,
        [userId]
      );

      const userRow = u[0] ?? null;

      return {
        user: {
          id: userId,
          name: userRow?.name ?? null,
          email: userRow?.email ?? null,
          username: userRow?.username ?? null,
          accessLevel: req.user?.accessLevel?.name ?? null,
        },
        preferences: {
          dashboardCases,
        },
      };
    });

    res.json(out);
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getMe", message: e.message });
  }
}

/**
 * ✅ Elfogad kétféle body-t:
 * 1) { dashboardCases: {...} }
 * 2) { q, status, mine, ..., viewMode }  // "flat"
 */
const SaveDashboardCasesBody = z
  .object({
    dashboardCases: z.record(z.any()).optional(),
  })
  .passthrough();

/**
 * Opcionális extra validáció:
 * ha viewMode érkezik, legyen "kanban" | "table"
 */
function validateDashboardCases(d: Record<string, any>) {
  if (d.viewMode !== undefined) {
    if (d.viewMode !== "kanban" && d.viewMode !== "table") {
      const err: any = new Error("Invalid viewMode (allowed: kanban | table)");
      err.statusCode = 400;
      throw err;
    }
  }
}

export async function saveDashboardCasesPrefs(req: Request, res: Response) {
  try {
    const b = SaveDashboardCasesBody.parse(req.body);

    // normalize: ha flat body, akkor maga a body a dashboardCases
    const dashboardCases =
      (b.dashboardCases as Record<string, any> | undefined) ??
      (req.body as Record<string, any>);

    // védjük ki, hogy véletlenül dashboardCases wrapperrel együtt merge-öljük
    if ("dashboardCases" in dashboardCases)
      delete (dashboardCases as any).dashboardCases;

    validateDashboardCases(dashboardCases);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const userId = req.user!.id;

      /**
       * ✅ Merge update:
       * - ha nincs row: insert
       * - ha van: meglévő jsonb || új jsonb
       *   (azaz a küldött kulcsok felülírják a régieket)
       */
      await db.query(
        `
        INSERT INTO public.user_preferences (user_id, dashboard_cases)
        VALUES ($1, $2::jsonb)
        ON CONFLICT (user_id)
        DO UPDATE SET dashboard_cases = COALESCE(public.user_preferences.dashboard_cases, '{}'::jsonb) || EXCLUDED.dashboard_cases
        `,
        [userId, JSON.stringify(dashboardCases ?? {})]
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "saveDashboardCasesPrefs", message: e.message });
  }
}
