// src/app/controllers/api/app/meetings.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";
import { requireCaseRole } from "../../../../utils/aclCase.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";
const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

async function resolveMeetingId(db: Db, meetingRef: string): Promise<string> {
  if (UUID_RE.test(meetingRef)) return z.string().uuid().parse(meetingRef);

  const r = await db.query(
    `SELECT id FROM public.meetings WHERE code=$1 LIMIT 1`,
    [meetingRef.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Meeting not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}
function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}
const AgendaItemWithCase = z.object({
  caseId: z.string().uuid(),
  threadId: z.string().uuid().optional(),
  title: z.string().min(2),
  sortOrder: z.number().int().optional(),
});

const AgendaItemNoCase = z.object({
  // "general" meeting agenda item (case nélkül)
  title: z.string().min(2),
  sortOrder: z.number().int().optional(),
});
const CreateMeetingBody = z.object({
  title: z.string().min(2),
  startsAt: z.string().datetime(),
  endsAt: z.string().datetime().optional(),
  notes: z.string().optional(),
  agendaItems: z
    .array(z.union([AgendaItemWithCase, AgendaItemNoCase]))
    .optional(),
});

export async function createMeeting(req: Request, res: Response) {
  try {
    const b = CreateMeetingBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const { rows } = await db.query<{ id: string; code: string }>(
        `
        INSERT INTO public.meetings (id, title, starts_at, ends_at, notes, created_by)
        VALUES (gen_random_uuid(), $1, $2, $3, $4, $5)
        RETURNING id, code
        `,
        [
          b.title.trim(),
          new Date(b.startsAt),
          b.endsAt ? new Date(b.endsAt) : null,
          b.notes ?? null,
          req.user!.id,
        ]
      );

      const meetingId = rows[0].id;

      if (b.agendaItems?.length) {
        for (const it of b.agendaItems) {
          // case nélküli agenda item: most csak skip (meeting megvan)
          if (!("caseId" in it)) continue;

          await requireCaseRole(db, it.caseId, "editor");

          await db.query(
            `
      INSERT INTO public.meeting_agenda_items
        (id, meeting_id, case_id, thread_id, title, sort_order)
      VALUES
        (gen_random_uuid(), $1, $2, $3, $4, $5)
      `,
            [
              meetingId,
              it.caseId,
              ("threadId" in it ? it.threadId : undefined) ?? null,
              it.title.trim(),
              it.sortOrder ?? 0,
            ]
          );
        }
      }

      return rows[0];
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_CREATED",
      objectType: "meeting",
      objectId: created.id,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { title: b.title, starts_at: b.startsAt, code: created.code },
    });

    res.status(201).json({ id: created.id, code: created.code });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createMeeting", message: e.message });
  }
}

const ListMeetingsQuery = z.object({
  from: z.string().datetime().optional(),
  to: z.string().datetime().optional(),
  q: z.string().optional(),
  includeArchived: z.union([z.literal("1"), z.literal("true")]).optional(),
});

export async function listMeetings(req: Request, res: Response) {
  try {
    const q = ListMeetingsQuery.parse(req.query);

    const meetings = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const params: any[] = [];
      const where: string[] = [];

      if (!q.includeArchived) where.push(`m.archived_at IS NULL`);

      if (q.from) {
        params.push(new Date(q.from));
        where.push(`m.starts_at >= $${params.length}`);
      }
      if (q.to) {
        params.push(new Date(q.to));
        where.push(`m.starts_at < $${params.length}`);
      }
      if (q.q?.trim()) {
        params.push(`%${q.q.trim()}%`);
        where.push(
          `(m.code ILIKE $${params.length} OR m.title ILIKE $${params.length})`
        );
      }

      const sql = `
        SELECT
          m.id, m.code, m.title, m.starts_at, m.ends_at, m.notes,
          m.created_by, m.created_at,
          m.archived_at, m.archived_by
        FROM public.meetings m
        ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
        ORDER BY m.starts_at DESC
        LIMIT 200
      `;

      const { rows } = await db.query(sql, params);
      return rows;
    });

    res.json({ meetings });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listMeetings", message: e.message });
  }
}

// const ListMeetingsQuery = z.object({
//   from: z.string().datetime().optional(),
//   to: z.string().datetime().optional(),
// });
// export async function listMeetings(req: Request, res: Response) {
//   try {
//     const q = ListMeetingsQuery.parse(req.query);

//     const meetings = await withRlsTx(rlsCtx(req), async (db: Db) => {
//       const params: any[] = [];
//       const where: string[] = [];

//       if (q.from) {
//         params.push(new Date(q.from));
//         where.push(`m.starts_at >= $${params.length}`);
//       }
//       if (q.to) {
//         params.push(new Date(q.to));
//         where.push(`m.starts_at < $${params.length}`);
//       }

//       const sql = `
//         SELECT
//           m.id, m.title, m.starts_at, m.ends_at, m.notes,
//           m.created_by, m.created_at
//         FROM public.meetings m
//         ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
//         ORDER BY m.starts_at DESC
//         LIMIT 200
//       `;

//       const { rows } = await db.query(sql, params);
//       return rows;
//     });

//     res.json({ meetings });
//   } catch (e: any) {
//     res.status(400).json({ error: "listMeetings", message: e.message });
//   }
// }
export async function getMeetingDetail(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);

    const data = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      const { rows: m } = await db.query(
        `
        SELECT id, code, title, starts_at, ends_at, notes, created_by, created_at,
               archived_at, archived_by
        FROM public.meetings
        WHERE id=$1
        `,
        [meetingId]
      );
      if (!m.length) return null;

      const includeArchived =
        req.query.includeArchived === "1" ||
        req.query.includeArchived === "true";

      const { rows: agenda } = await db.query(
        `
  SELECT
    ai.id,
    ai.meeting_id,
    ai.case_id,
    ai.thread_id,
    ai.title,
    ai.sort_order,
    ai.archived_at,
    ai.archived_by,
    c.title AS case_title,
    c.code  AS case_code,
    t.title AS thread_title,
    t.code  AS thread_code
  FROM public.meeting_agenda_items ai
  JOIN public.cases c ON c.id = ai.case_id
  LEFT JOIN public.case_threads t ON t.id = ai.thread_id
  WHERE ai.meeting_id=$1
    AND ($2::boolean = true OR ai.archived_at IS NULL)
  ORDER BY ai.sort_order ASC, ai.title ASC
  `,
        [meetingId, includeArchived]
      );

      const { rows: attachments } = await db.query(
        `
        SELECT
          a.id,
          a.original_name,
          a.mime_type,
          a.size_bytes,
          a.uploaded_at,
          a.uploaded_by
        FROM public.meeting_attachments ma
        JOIN public.attachments a ON a.id = ma.attachment_id
        WHERE ma.meeting_id = $1
        ORDER BY a.uploaded_at DESC
        `,
        [meetingId]
      );

      return {
        meeting: m[0],
        agenda,
        attachments: attachments.map((r) => ({
          ...r,
          downloadUrl: `/api/app/attachments/${r.id}/download`,
        })),
      };
    });

    if (!data) return res.status(404).json({ error: "Not found" });
    res.json(data);
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getMeetingDetail", message: e.message });
  }
}

const UpdateMeetingBody = z.object({
  title: z.string().min(2).optional(),
  startsAt: z.string().datetime().optional(),
  endsAt: z.string().datetime().optional(),
  notes: z.string().optional(),
});
export async function updateMeeting(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);
    const b = UpdateMeetingBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      const r = await db.query(
        `SELECT created_by, archived_at FROM public.meetings WHERE id=$1`,
        [meetingId]
      );
      const cur = (r.rows ?? []) as Array<{
        created_by: string | null;
        archived_at: string | null;
      }>;
      if (!cur.length) {
        const e: any = new Error("Meeting not found");
        e.statusCode = 404;
        throw e;
      }

      if (cur[0].created_by && cur[0].created_by !== req.user!.id) {
        // ✅ creator nem ő -> legyen owner/editor legalább egy agenda item case-ében
        const { rows: ok } = await db.query<{ ok: boolean }>(
          `
    SELECT EXISTS (
      SELECT 1
      FROM public.meeting_agenda_items ai
      JOIN public.case_members m
        ON m.case_id = ai.case_id
      WHERE ai.meeting_id = $1
        AND m.user_id = public.app_user_id()
        AND m.is_active = true
        AND m.role IN ('owner','editor')
    ) AS ok
    `,
          [meetingId]
        );

        if (!ok[0]?.ok) {
          const e: any = new Error(
            "Nincs jogosultság a meeting módosításához."
          );
          e.statusCode = 403;
          throw e;
        }
      }

      const hasTitle = b.title !== undefined;
      const hasStartsAt = b.startsAt !== undefined;
      const hasEndsAt = b.endsAt !== undefined;
      const hasNotes = b.notes !== undefined;

      await db.query(
        `
  UPDATE public.meetings
  SET
    title = CASE WHEN $2::boolean THEN $3 ELSE title END,
    starts_at = CASE WHEN $4::boolean THEN $5 ELSE starts_at END,
    ends_at = CASE WHEN $6::boolean THEN $7 ELSE ends_at END,
    notes = CASE WHEN $8::boolean THEN $9 ELSE notes END,
    updated_by = $10,
    updated_at = now()
  WHERE id = $1
  `,
        [
          meetingId,

          // title
          hasTitle,
          hasTitle ? b.title!.trim() : null,

          // starts_at
          hasStartsAt,
          hasStartsAt ? new Date(b.startsAt!) : null,

          // ends_at (nullable)
          hasEndsAt,
          hasEndsAt ? (b.endsAt ? new Date(b.endsAt) : null) : null,

          // notes (nullable)
          hasNotes,
          hasNotes ? b.notes ?? null : null,

          req.user!.id,
        ]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_UPDATED",
      objectType: "meeting",
      objectId: meetingRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { meeting_ref: meetingRef, changed: Object.keys(b) },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "updateMeeting", message: e.message });
  }
}

// export async function updateMeeting(req: Request, res: Response) {
//   try {
//     const meetingId = z.string().uuid().parse(req.params.meetingId);
//     const b = UpdateMeetingBody.parse(req.body);

//     await withRlsTx(rlsCtx(req), async (db: Db) => {
//       // NOTE: ehhez DB oldalon UPDATE policy kell a public.meetings táblára
//       const { rows } = await db.query(
//         `
//         UPDATE public.meetings
//         SET
//           title = COALESCE($2, title),
//           starts_at = COALESCE($3, starts_at),
//           ends_at = COALESCE($4, ends_at),
//           notes = COALESCE($5, notes)
//         WHERE id=$1
//         `,
//         [
//           meetingId,
//           b.title ? b.title.trim() : null,
//           b.startsAt ? new Date(b.startsAt) : null,
//           b.endsAt ? new Date(b.endsAt) : null,
//           b.notes ?? null,
//         ]
//       );

//       // rows nincs RETURNING, de elég
//       void rows;
//     });

//     void writeAuditLog({
//       actorId: req.user?.id ?? null,
//       action: "MEETING_UPDATED",
//       objectType: "meeting",
//       objectId: meetingId,
//       route: req.originalUrl,
//       ip: req.ip,
//       payloadHash: null,
//       meta: { changed: Object.keys(b) },
//     });

//     res.json({ ok: true });
//   } catch (e: any) {
//     res.status(400).json({ error: "updateMeeting", message: e.message });
//   }
// }

const AddAgendaItemBody = z.object({
  caseId: z.string().uuid(),
  threadId: z.string().uuid().optional(),
  title: z.string().min(2),
  sortOrder: z.number().int().optional(),
});

export async function addAgendaItem(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);
    const b = AddAgendaItemBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      await requireCaseRole(db, b.caseId, "editor");

      const { rows } = await db.query<{ id: string }>(
        `
        INSERT INTO public.meeting_agenda_items
          (id, meeting_id, case_id, thread_id, title, sort_order)
        VALUES
          (gen_random_uuid(), $1, $2, $3, $4, $5)
        RETURNING id
        `,
        [
          meetingId,
          b.caseId,
          b.threadId ?? null,
          b.title.trim(),
          b.sortOrder ?? 0,
        ]
      );

      return rows[0].id;
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_AGENDA_ITEM_ADDED",
      objectType: "meeting",
      objectId: meetingRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {
        meeting_ref: meetingRef,
        agenda_item_id: created,
        case_id: b.caseId,
        thread_id: b.threadId,
      },
    });

    res.status(201).json({ id: created });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "addAgendaItem", message: e.message });
  }
}

const PatchAgendaItemBody = z.object({
  title: z.string().min(2).optional(),
  sortOrder: z.number().int().optional(),
  threadId: z.string().uuid().nullable().optional(),
});

export async function patchAgendaItem(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);
    const agendaItemId = z.string().uuid().parse(req.params.agendaItemId);
    const b = PatchAgendaItemBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      const { rows: ai } = await db.query<{ case_id: string }>(
        `SELECT case_id FROM public.meeting_agenda_items WHERE id=$1 AND meeting_id=$2`,
        [agendaItemId, meetingId]
      );
      if (!ai.length) {
        const e: any = new Error("Agenda item not found");
        e.statusCode = 404;
        throw e;
      }

      await requireCaseRole(db, ai[0].case_id, "editor");

      await db.query(
        `
        UPDATE public.meeting_agenda_items
        SET
          title = COALESCE($3, title),
          sort_order = COALESCE($4, sort_order),
          thread_id = COALESCE($5, thread_id)
        WHERE id=$1 AND meeting_id=$2
        `,
        [
          agendaItemId,
          meetingId,
          b.title ? b.title.trim() : null,
          b.sortOrder ?? null,
          b.threadId === undefined ? null : b.threadId,
        ]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_AGENDA_ITEM_UPDATED",
      objectType: "meeting",
      objectId: meetingRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {
        meeting_ref: meetingRef,
        agenda_item_id: agendaItemId,
        changed: Object.keys(b),
      },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchAgendaItem", message: e.message });
  }
}

export async function deleteAgendaItem(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);
    const agendaItemId = z.string().uuid().parse(req.params.agendaItemId);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      const { rows: ai } = await db.query<{ case_id: string }>(
        `SELECT case_id FROM public.meeting_agenda_items WHERE id=$1 AND meeting_id=$2`,
        [agendaItemId, meetingId]
      );
      if (!ai.length) {
        const e: any = new Error("Agenda item not found");
        e.statusCode = 404;
        throw e;
      }

      await requireCaseRole(db, ai[0].case_id, "editor");

      await db.query(
        `DELETE FROM public.meeting_agenda_items WHERE id=$1 AND meeting_id=$2`,
        [agendaItemId, meetingId]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_AGENDA_ITEM_DELETED",
      objectType: "meeting",
      objectId: meetingRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { meeting_ref: meetingRef, agenda_item_id: agendaItemId },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "deleteAgendaItem", message: e.message });
  }
}

export async function archiveAgendaItem(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);
    const agendaItemId = z.string().uuid().parse(req.params.agendaItemId);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      const r = await db.query(
        `SELECT case_id FROM public.meeting_agenda_items WHERE id=$1 AND meeting_id=$2`,
        [agendaItemId, meetingId]
      );
      const ai = (r.rows ?? []) as Array<{ case_id: string }>;
      if (!ai.length) {
        const e: any = new Error("Agenda item not found");
        e.statusCode = 404;
        throw e;
      }

      await requireCaseRole(db, ai[0].case_id, "editor");

      await db.query(
        `
        UPDATE public.meeting_agenda_items
        SET archived_at=now(), archived_by=$3
        WHERE id=$1 AND meeting_id=$2 AND archived_at IS NULL
        `,
        [agendaItemId, meetingId, req.user!.id]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_AGENDA_ITEM_ARCHIVED",
      objectType: "meeting",
      objectId: meetingRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { meeting_ref: meetingRef, agenda_item_id: agendaItemId },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "archiveAgendaItem", message: e.message });
  }
}

export async function archiveMeeting(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      const r = await db.query(
        `SELECT created_by, archived_at FROM public.meetings WHERE id=$1`,
        [meetingId]
      );
      if (!r.rows?.length) {
        const e: any = new Error("Meeting not found");
        e.statusCode = 404;
        throw e;
      }

      // author only (mint updateMeeting)
      if (r.rows[0].created_by && r.rows[0].created_by !== req.user!.id) {
        const e: any = new Error("Nincs jogosultság a meeting archiválásához.");
        e.statusCode = 403;
        throw e;
      }

      await db.query(
        `
        UPDATE public.meetings
        SET archived_at=now(), archived_by=$2, updated_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        [meetingId, req.user!.id]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "MEETING_ARCHIVED",
      objectType: "meeting",
      objectId: meetingRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { meeting_ref: meetingRef },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "archiveMeeting", message: e.message });
  }
}
export async function unarchiveAgendaItem(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);
    const agendaItemId = z.string().uuid().parse(req.params.agendaItemId);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      const { rows: ai } = await db.query<{ case_id: string }>(
        `SELECT case_id FROM public.meeting_agenda_items WHERE id=$1 AND meeting_id=$2`,
        [agendaItemId, meetingId]
      );
      if (!ai.length) {
        const e: any = new Error("Agenda item not found");
        e.statusCode = 404;
        throw e;
      }

      await requireCaseRole(db, ai[0].case_id, "editor");

      await db.query(
        `
        UPDATE public.meeting_agenda_items
        SET archived_at=NULL, archived_by=NULL
        WHERE id=$1 AND meeting_id=$2 AND archived_at IS NOT NULL
        `,
        [agendaItemId, meetingId]
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "unarchiveAgendaItem", message: e.message });
  }
}
export async function unarchiveMeeting(req: Request, res: Response) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);

      // ugyanaz a logika, mint archiveMeeting-ben (creator vagy case owner/editor az agenda alapján)
      const r = await db.query(
        `SELECT created_by, archived_at FROM public.meetings WHERE id=$1`,
        [meetingId]
      );
      if (!r.rows?.length) {
        const e: any = new Error("Meeting not found");
        e.statusCode = 404;
        throw e;
      }

      if (r.rows[0].created_by && r.rows[0].created_by !== req.user!.id) {
        const { rows: ok } = await db.query<{ ok: boolean }>(
          `
          SELECT EXISTS (
            SELECT 1
            FROM public.meeting_agenda_items ai
            JOIN public.case_members m ON m.case_id = ai.case_id
            WHERE ai.meeting_id = $1
              AND m.user_id = public.app_user_id()
              AND m.is_active = true
              AND m.role IN ('owner','editor')
          ) AS ok
          `,
          [meetingId]
        );

        if (!ok[0]?.ok) {
          const e: any = new Error(
            "Nincs jogosultság a meeting visszaállításához."
          );
          e.statusCode = 403;
          throw e;
        }
      }

      await db.query(
        `
        UPDATE public.meetings
        SET
          archived_at = NULL,
          archived_by = NULL,
          updated_by = $2,
          updated_at = now()
        WHERE id=$1
        `,
        [meetingId, req.user!.id]
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "unarchiveMeeting", message: e.message });
  }
}
